
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  Alert,
  Modal,
  Image,
  Animated,
  Share,
  Linking,
  TextInput,
  KeyboardAvoidingView,
  Platform
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import SignatureCanvas from 'react-native-signature-canvas';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import Toast from 'react-native-toast-message';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width, height } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_020Step3 = ({ setActiveStep }) => {
  console.log('HSE_020Step3 component rendered');
  const navigation = useNavigation();
  const [step2Data, setStep2Data] = useState(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const isMounted = useRef(true);
  
  // Signature modal states
  const [signatureModalVisible, setSignatureModalVisible] = useState(false);
  const [currentSignatureField, setCurrentSignatureField] = useState('');
  const [signatures, setSignatures] = useState({
    observed_person: {
      signatureData: null,
      signatureStatus: false
    },
    conducting_observation_and_leadership: {
      name: '',
      signatureData: null,
      signatureStatus: false
    }
  });

  const signatureRef = useRef(null);
  const capturedSignature = useRef(null);

  useEffect(() => {
    loadData();
    isMounted.current = true;
    return () => {
      isMounted.current = false;
    };
  }, []);

  // MongoDB helper functions
  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  // MongoDB helper functions for report numbering
  const getCurrentReportNumber = async () => {
    try {
      const response = await HSEInspectionAPI.getCurrentCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || response.data.nextNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error getting current report number from MongoDB:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return '001';
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId, onProgress) => {
    try {
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_020_${userId}_${Date.now()}.pdf`
      };

      const response = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      
      if (!response.success) {
        throw new Error('Failed to upload PDF to MongoDB');
      }

      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading file to MongoDB:', error);
      throw error;
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseFinalReport');
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const loadData = async () => {
    try {
      setIsLoading(true);
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        const data = JSON.parse(savedStep2);
        setStep2Data(data);
        // Load existing signatures if any
        if (data.signatures) {
          setSignatures(data.signatures);
        }
      }
    } catch (error) {
      console.error('Error loading data:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const openSignatureModal = (field) => {
    setCurrentSignatureField(field);
    setSignatureModalVisible(true);
    capturedSignature.current = null;
  };

  const closeSignatureModal = () => {
    setSignatureModalVisible(false);
    setCurrentSignatureField('');
  };

  const handleSignatureComplete = (signature) => {
    if (signature) {
      setSignatures(prev => ({
        ...prev,
        [currentSignatureField]: {
          ...prev[currentSignatureField],
          signatureData: signature,
          signatureStatus: true
        }
      }));
    }
    setSignatureModalVisible(false);
  };

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      capturedSignature.current = null;
    }
  };

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    }
  };

  const handleSignatureData = (signature) => {
    console.log('Signature captured:', signature ? signature.substring(0, 50) + '...' : signature);
    capturedSignature.current = signature;
    handleSignatureComplete(signature);
  };

  const validateForm = () => {
    // Check if signatures are filled
    if (!signatures.observed_person.signatureStatus || !signatures.conducting_observation_and_leadership.signatureStatus) {
      Alert.alert('Missing Signatures', 'Please complete both signature fields before submitting.');
      return false;
    }
    
    // Check if conducting observation name is filled
    if (!signatures.conducting_observation_and_leadership.name.trim()) {
      Alert.alert('Missing Name', 'Please enter the name of person conducting observation and leadership.');
      return false;
    }
    
    return true;
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    setIsLoading(true);
    setUploadProgress(0);

    try {
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      
      if (!savedStep1 || !savedStep2) {
        throw new Error('Previous step data is missing');
      }

      const step1Data = JSON.parse(savedStep1);
      const step2Data = JSON.parse(savedStep2);

      // Use the request number from step 1
      const requestId = step1Data.requestNumber;
      if (!requestId) {
        throw new Error('Request number not found in step 1 data');
      }
      setReportId(requestId);
      setUploadProgress(20);

      // Generate PDF
      console.log('📄 Generating PDF...');
      const reportData = {
        id: requestId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: signatures,
      };

      const pdfPath = await generatePDFWithHtml(reportData);
      console.log('✅ PDF generated at:', pdfPath);
      setUploadProgress(50);

      // Upload PDF to MongoDB
      console.log('📤 Uploading PDF to MongoDB...');
      const pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId, (progress) => {
        const pdfProgress = 50 + (progress / 2);
        if (isMounted.current) {
          setUploadProgress(pdfProgress);
        }
      });
      console.log('✅ PDF uploaded:', pdfDownloadUrl);
      setUploadProgress(80);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      const completeReportData = {
        _id: requestId,
        userId,
        requestNumber: requestId,
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: signatures,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      const saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
      
      if (!saveResponse.success) {
        throw new Error('Failed to save report to MongoDB');
      }
      
      console.log('✅ Report saved to MongoDB');
      setUploadProgress(90);

      // Delete temp PDF
      await RNFS.unlink(pdfPath).catch((err) => console.warn('⚠️ Error deleting temp PDF:', err));
      
      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: requestId,
            id: requestId,
            requestNumber: requestId,
            checklistName: 'HSE-020 Task Observation and Leadership',
            country: step1.country || '',
            project: step1.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('https://api.titandrillingzm.com:6028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }
      
      // Increment counter
      await incrementReportCounter();
      setUploadProgress(100);
      
      // Clear local data
      await flushStepData(requestId);

      if (isMounted.current) {
        setIsLoading(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Task Observation Report submitted to MongoDB successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('❌ Error in handleSubmit:', error);
      if (isMounted.current) {
        setIsLoading(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to submit report to MongoDB.',
          position: 'top',
        });
        Alert.alert('Submit Error', `Failed to submit report to MongoDB.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      // Use the same database path as Hsemodule43step3
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      navigation.navigate('HSEViewPDF', { 
        pdfUrl: pdfUrl, 
        onGoBack: () => { 
          setShowSuccessModal(false); 
        } 
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};
      const checklist = report.checklist || {};
      const pdfData = report.pdfData || {}; // Get pdfData from report

      // Generate evaluation criteria rows
      let evaluationRows = '';
      const evaluationCriteria = step2.evaluation_criteria || {};
      const criteriaQuestions = [
        'Could any of the practices or conditions observed result in property damage or personal injury?',
        'Are the methods and practices observed the most efficient and productive for the task under observation?',
        'Did the practices observed comply with all of the applicable standards that exist for the task or job?',
        'Could any of the practices observed have a detrimental effect upon the quality of the work undertaken?'
      ];
      const criteriaFields = [
        'property_damage_or_personal_injury',
        'efficient_and_productive',
        'comply_with_standards',
        'detrimental_effect_on_quality'
      ];

      criteriaQuestions.forEach((question, index) => {
        const field = criteriaFields[index];
        const criteria = evaluationCriteria[field] || {};
        const yesChecked = criteria.yes ? '✓' : '';
        const noChecked = criteria.no ? '✓' : '';
        
        evaluationRows += `
          <tr>
            <td>${question}</td>
            <td style="text-align: center;">${yesChecked}</td>
            <td style="text-align: center;">${noChecked}</td>
          </tr>
        `;
      });

      // Generate correction section rows
      let correctionRows = '';
      const correctionSection = step2.correction_section || {};
      const deviationsCorrections = correctionSection.deviations_corrections || [];
      
      deviationsCorrections.forEach((item, index) => {
        if (item.deviation || item.correction) {
          correctionRows += `
            <tr>
              <td>${item.deviation || ''}</td>
              <td>${item.correction || ''}</td>
            </tr>
          `;
        }
      });

      // Generate action items
      let actionItemsHtml = '';
      const actionItems = step2.action_items || {};
      
      const actionQuestions = [
        'Has the employee been complimented / instructed based on the observations:',
        'Should a follow up observation be conducted on the person(s) conducting this particular task:'
      ];
      const actionFields = [
        'employee_complimented_instructed',
        'follow_up_observation_needed'
      ];

      actionQuestions.forEach((question, index) => {
        const field = actionFields[index];
        const action = actionItems[field] || {};
        const yesChecked = action.yes ? '✓' : '';
        const noChecked = action.no ? '✓' : '';
        
        actionItemsHtml += `
          <tr>
            <td>${question}</td>
            <td style="text-align: center;">${yesChecked}</td>
            <td style="text-align: center;">${noChecked}</td>
          </tr>
        `;
      });

      // Generate signature images
      const observedSignature = signatures.observed_person.signatureStatus && signatures.observed_person.signatureData 
        ? `<img src="${signatures.observed_person.signatureData}" alt="Observed Person Signature" style="width: 150px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
        : '';
      
      const conductingSignature = signatures.conducting_observation_and_leadership.signatureStatus && signatures.conducting_observation_and_leadership.signatureData 
        ? `<img src="${signatures.conducting_observation_and_leadership.signatureData}" alt="Conducting Observation Signature" style="width: 150px; height: 60px; object-fit: contain; border: 1px solid #ccc;"/>`
        : '';

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>${checklist.document_name || 'HSE-020 Task Observation and Leadership'}</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
              padding: 0;
              font-size: 12px;
            }
            .header-row {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 10px;
            }
            .logo-left {
              display: flex;
              align-items: flex-start;
            }
            .logo-left img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .doc-info-box {
              width: 340px;
            }
            .doc-info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 12px;
            }
            .doc-info-box td {
              border: 1px solid #222;
              padding: 4px 8px;
              text-align: left;
            }
            .title-section {
              text-align: center;
              margin: 10px 0 0 0;
            }
            .title-section h2 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              text-decoration: underline;
            }
            .content-section {
              margin: 15px 0;
            }
            .content-section h3 {
              font-size: 14px;
              font-weight: bold;
              margin: 10px 0 5px 0;
            }
            .content-section ul {
              margin: 5px 0;
              padding-left: 20px;
            }
            .content-section li {
              margin: 2px 0;
            }
            .observation-table {
              width: 100%;
              border-collapse: collapse;
              margin: 10px 0;
            }
            .observation-table th,
            .observation-table td {
              border: 1px solid #000;
              padding: 5px;
              text-align: left;
            }
            .observation-table th {
              background-color: #f2f2f2;
              font-weight: bold;
            }
            .signature-section {
              display: flex;
              justify-content: space-between;
              margin-top: 20px;
            }
            .signature-box {
              width: 48%;
              border: 1px solid #000;
              padding: 10px;
              min-height: 80px;
            }
            .signature-label {
              font-weight: bold;
              margin-bottom: 10px;
            }
            .signature-image {
              text-align: center;
              margin-top: 10px;
            }
            .checkbox-item {
              margin: 5px 0;
            }
            .checkbox-item input[type="checkbox"] {
              margin-right: 5px;
            }
            .text-area {
              border: 1px solid #000;
              padding: 5px;
              min-height: 60px;
              margin: 5px 0;
            }
          </style>
        </head>
        <body>
          <div class="header-row">
            <div class="logo-left">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
            </div>
            <div class="doc-info-box">
              <table>
                <tr><td>Document Name</td><td>${checklist.document_name || 'HSE-FOR-8.1.1-Task Observation and Leadership'}</td></tr>
                <tr><td>Document ID</td><td>${checklist.document_id || 'HSE-020'}</td></tr>
                <tr><td>Version Date</td><td>${checklist.version_date || '14/7/2024'}</td></tr>
                <tr><td>Version</td><td>${checklist.version || '2'}</td></tr>
                <tr><td>Approved</td><td>${checklist.approved_by || 'HSM'}</td></tr>
                <tr><td>Request #</td><td>${pdfData.request_number || 'N/A'}</td></tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h2>TASK OBSERVATION AND LEADERSHIP</h2>
          </div>

          <div class="content-section">
            <h3>Observation Details</h3>
            <table class="observation-table">
              <tr>
                <td><strong>Report No:</strong></td>
                <td>${report.id || 'N/A'}</td>
              </tr>
              <tr>
                <td><strong>Country:</strong></td>
                <td>${step1.country || 'N/A'}</td>
              </tr>
              <tr>
                <td><strong>Project:</strong></td>
                <td>${step1.project || 'N/A'}</td>
              </tr>
              <tr>
                <td><strong>Name(s) of person(s) under observation:</strong></td>
                <td>${step2.name_of_person_under_observation || ''}</td>
              </tr>
              <tr>
                <td><strong>Date:</strong></td>
                <td>${step2.date || ''}</td>
              </tr>
              <tr>
                <td><strong>Department:</strong></td>
                <td>${step2.department || ''}</td>
              </tr>
              <tr>
                <td><strong>Task under observation:</strong></td>
                <td>${step2.task_under_observation || ''}</td>
              </tr>
            </table>
          </div>

          <div class="content-section">
            <h3>Type of observation</h3>
            <div class="checkbox-item">
              <input type="checkbox" ${step2.type_of_observation?.planned_task_observation ? 'checked' : ''} disabled> Planned Task Observation
            </div>
            <div class="checkbox-item">
              <input type="checkbox" ${step2.type_of_observation?.visible_felt_leadership ? 'checked' : ''} disabled> Visible Felt Leadership
            </div>
            <div style="text-align: right; margin-top: 10px;">
              <strong>Approved by: HSM</strong>
            </div>
          </div>

          <div class="content-section">
            <h3>Reason for PTO/VFL</h3>
            <div style="display: flex; justify-content: space-between;">
              <div style="width: 48%;">
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.task_procedure_review ? 'checked' : ''} disabled> TASK PROCEDURE REVIEW
                </div>
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.behavioural ? 'checked' : ''} disabled> BEHAVIOURAL
                </div>
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.training_follow_up ? 'checked' : ''} disabled> TRAINING FOLLOW UP
                </div>
              </div>
              <div style="width: 48%;">
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.incident_occurred ? 'checked' : ''} disabled> INCIDENT OCCURRED
                </div>
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.new_employee ? 'checked' : ''} disabled> NEW EMPLOYEE
                </div>
                <div class="checkbox-item">
                  <input type="checkbox" ${step2.reason_for_pto_vfl?.equipment_suitability ? 'checked' : ''} disabled> EQUIPMENT SUITABILITY
                </div>
              </div>
            </div>
          </div>

          <div class="content-section">
            <h3>Evaluation Criteria</h3>
            <table class="observation-table">
              <tr>
                <th>Question</th>
                <th>YES</th>
                <th>NO</th>
              </tr>
              ${evaluationRows}
            </table>
          </div>

          <div class="content-section">
            <h3>Commendation Section</h3>
            <div style="margin-bottom: 10px; font-weight: bold; color: #015185;">
              Describe clearly below any practices or conditions related to observation that deserve commendation
            </div>
            <div class="text-area">
              ${step2.commendation_section?.details || ''}
            </div>
          </div>

          <div class="content-section">
            <h3>Correction Section</h3>
            <div style="margin-bottom: 10px; font-weight: bold; color: #015185;">
              Describe clearly below any practices or conditions related to observation that deserve correction
            </div>
            <table class="observation-table">
              <tr>
                <th>Deviation</th>
                <th>Correction</th>
              </tr>
              ${correctionRows}
            </table>
          </div>

          <div class="content-section">
            <h3>Follow-up Actions</h3>
            <table class="observation-table">
              <tr>
                <th>Question</th>
                <th>YES</th>
                <th>NO</th>
              </tr>
              ${actionItemsHtml}
            </table>
          </div>

          <div class="signature-section">
            <div class="signature-box">
              <div class="signature-label">Signature(s) of observed person(s):</div>
              <div class="signature-image">
                ${observedSignature}
              </div>
            </div>
            <div class="signature-box">
              <div class="signature-label">Name & Signature of person(s) conducting observation and leadership:</div>
              <div style="margin-bottom: 5px;"><strong>Name:</strong> ${signatures.conducting_observation_and_leadership.name || ''}</div>
              <div class="signature-image">
                ${conductingSignature}
              </div>
            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `HSE_020_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const canvasStyle = `
    .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
    .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
    canvas { height: 100% !important; width: 100% !important; }
  `;

  if (isLoading) {
    return (
      <View style={styles.container}>
        <SyncModal visible={isLoading} message="Generating Report" />
      </View>
    );
  }

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>

        {/* Observation Name */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Observation Details</Text>
          <View style={styles.infoRow}>
            <Text style={styles.infoLabel}>Person Under Observation:</Text>
            <Text style={styles.infoValue}>
              {step2Data?.name_of_person_under_observation || 'N/A'}
            </Text>
          </View>
        </View>

        {/* Signatures Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Signatures</Text>
          
          {/* Observed Person Signature */}
          <View style={styles.signatureField}>
            <Text style={styles.signatureLabel}>Signature(s) of observed person(s):</Text>
            {signatures.observed_person.signatureStatus && signatures.observed_person.signatureData ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: signatures.observed_person.signatureData }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('observed_person')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('observed_person')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>

          {/* Conducting Observation Signature */}
          <View style={styles.signatureField}>
            <Text style={styles.signatureLabel}>Name & Signature of person(s) conducting observation and leadership:</Text>
            
            {/* Name Input */}
            <View style={styles.nameInputContainer}>
              <Text style={styles.nameInputLabel}>Name of conducting observation and leadership:</Text>
              <TextInput
                style={styles.nameInput}
                value={signatures.conducting_observation_and_leadership.name}
                onChangeText={(text) => setSignatures(prev => ({
                  ...prev,
                  conducting_observation_and_leadership: {
                    ...prev.conducting_observation_and_leadership,
                    name: text
                  }
                }))}
                placeholder="Enter name of person conducting observation"
              />
            </View>
            
            {/* Signature */}
            {signatures.conducting_observation_and_leadership.signatureStatus && signatures.conducting_observation_and_leadership.signatureData ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: signatures.conducting_observation_and_leadership.signatureData }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('conducting_observation_and_leadership')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('conducting_observation_and_leadership')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>
        </View>

        {/* Submit Section */}
        <View style={styles.submitSection}>
          <Text style={styles.submitTitle}>Ready to Submit</Text>
          <Text style={styles.submitDescription}>
            Please ensure both signatures are completed before submitting. Once submitted, this report will be saved and cannot be modified.
          </Text>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, isLoading && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={isLoading}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* SyncModal for Loading */}
      <SyncModal visible={isLoading} message="Generating Report" />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE-020 Task Observation and Leadership Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
                              <TouchableOpacity
                  style={styles.successButton}
                  onPress={() => {
                    setShowSuccessModal(false);
                    navigation.navigate('HSEInspectionHome');
                  }}
                >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Signature Modal with SignatureCanvas */}
      <Modal
        visible={signatureModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>
                {currentSignatureField === 'observed_person' 
                  ? 'Signature of Observed Person' 
                  : 'Signature of Conducting Observation'}
              </Text>
              <TouchableOpacity onPress={closeSignatureModal}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty');
                  capturedSignature.current = null;
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={canvasStyle}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error);
                  Alert.alert('Error', 'Failed to load signature canvas.');
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  section: {
    marginBottom: width * 0.04,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
    paddingVertical: 5,
  },
  infoLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    flex: 1,
  },
  infoValue: {
    fontSize: 16,
    color: '#666',
    flex: 2,
    textAlign: 'right',
  },
  signatureField: {
    marginBottom: 20,
  },
  signatureLabel: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
  },
  signatureImage: {
    width: width * 0.5,
    height: height * 0.12,
    borderRadius: 10,
    backgroundColor: '#f5f5f5',
  },
  resignButton: {
    backgroundColor: '#015185',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
  },
  resignButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  signatureBox: {
    height: height * 0.15,
    backgroundColor: '#f5f5f5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ddd',
    borderStyle: 'dashed',
  },
  signaturePlaceholder: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  nameInputContainer: {
    marginBottom: 15,
  },
  nameInputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 5,
  },
  nameInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  submitSection: {
    backgroundColor: '#e8f5e8',
    margin: 15,
    padding: 20,
    borderRadius: 8,
    borderWidth: 2,
    borderColor: '#4caf50',
  },
  submitTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#2e7d32',
    marginBottom: 10,
  },
  submitDescription: {
    fontSize: 14,
    color: '#2e7d32',
    lineHeight: 20,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 5,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  disabledButton: {
    opacity: 0.5,
  },

  // Modal Styles
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '95%',
    maxHeight: '90%',
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '60%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
});

export default HSE_020Step3;