
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Platform,
  StatusBar,
  Modal,
  Animated,
  BackHandler,
} from 'react-native';
import Pdf from 'react-native-pdf';
import LinearGradient from 'react-native-linear-gradient';
import RNFS from 'react-native-fs';
import Toast from 'react-native-toast-message';
import { PermissionsAndroid, Share } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import ReactNativeBlobUtil from 'react-native-blob-util'; // Import react-native-blob-util
import { useFocusEffect } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;
const TOTAL_HEADER_HEIGHT = width * 0.25;

const HSEViewPDF = ({ route, navigation }) => {
  const { pdfUrl, onGoBack } = route.params || {};
  const [localPdfPath, setLocalPdfPath] = useState(null);
  const [loading, setLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    console.log('PDF URL:', pdfUrl); // Debug the URL
    if (pdfUrl && (pdfUrl.startsWith('http://') || pdfUrl.startsWith('https://'))) {
      downloadPdfForViewing();
    }
    return () => {
      if (localPdfPath) {
        RNFS.unlink(localPdfPath).catch((err) => console.log('Error deleting temp file:', err));
      }
    };
  }, [pdfUrl]);

  // Handle back button and swipe back gestures
  useFocusEffect(
    React.useCallback(() => {
      const onBackPress = () => {
        try {
          navigation.navigate('HSEInspectionHome');
          return true; // Prevent default back behavior
        } catch (error) {
          console.log('Navigation error:', error);
          // Fallback to goBack if navigation fails
          navigation.goBack();
          return true;
        }
      };

      const backHandler = BackHandler.addEventListener('hardwareBackPress', onBackPress);

      // Handle swipe back gestures
      const unsubscribe = navigation.addListener('beforeRemove', (e) => {
        // Prevent default behavior
        e.preventDefault();
        
        try {
          navigation.navigate('HSEInspectionHome');
        } catch (error) {
          console.log('Navigation error on swipe back:', error);
          // If navigation fails, allow the default behavior
          navigation.dispatch(e.data.action);
        }
      });

      return () => {
        backHandler.remove();
        unsubscribe();
      };
    }, [navigation])
  );

  useEffect(() => {
    if (loading) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [loading]);

  const requestStoragePermission = async () => {
    if (Platform.OS !== 'android') return true;

    try {
      if (Platform.Version >= 33) {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.READ_MEDIA_IMAGES,
          {
            title: 'Media Permission Required',
            message: 'This app needs access to your media to view and download PDFs.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } else {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.WRITE_EXTERNAL_STORAGE,
          {
            title: 'Storage Permission Required',
            message: 'This app needs access to your storage to download the PDF.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          }
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      }
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const downloadPdfForViewing = async () => {
    if (!pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    setLoading(true);
    try {
      const hasPermission = await requestStoragePermission();
      if (!hasPermission) {
        throw new Error('Storage permission denied');
      }

      const tempPath = `${RNFS.DocumentDirectoryPath}/temp_report_${Date.now()}.pdf`;
      console.log('Attempting to download PDF to:', tempPath);

      const options = {
        fromUrl: pdfUrl,
        toFile: tempPath,
      };
      const result = await RNFS.downloadFile(options).promise;
      if (result.statusCode === 200) {
        setLocalPdfPath(tempPath);
        console.log('PDF downloaded for viewing to:', tempPath);
      } else {
        throw new Error(`Failed to download PDF for viewing. Status code: ${result.statusCode}`);
      }
    } catch (error) {
      console.error('Error downloading PDF for viewing:', error);
      let errorMessage = 'Failed to download PDF for viewing.';
      if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to view the PDF.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else {
        errorMessage = `Internal storage error: ${error.message}`;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 3000,
      });
      navigation.goBack();
    } finally {
      setLoading(false);
    }
  };

  const handleDownloadPDF = async () => {
    if (!pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF URL found.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Invalid PDF URL.',
        position: 'bottom',
        visibilityTime: 3000,
      });
      return;
    }

    try {
      const hasPermission = await requestStoragePermission();
      if (!hasPermission && Platform.OS === 'android') {
        throw new Error('Storage permission denied');
      }

      const fileName = `report_${Date.now()}.pdf`;
      let downloadDest;

      if (Platform.OS === 'android') {
        // On Android, use Download Manager to save to Downloads folder
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DownloadDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
          addAndroidDownloads: {
            useDownloadManager: true, // Use Android Download Manager
            notification: true, // Show notification
            title: fileName,
            description: 'Downloading PDF...',
            mime: 'application/pdf',
            mediaScannable: true, // Make file visible in Downloads app
          },
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', pdfUrl);
        console.log('PDF downloaded to:', res.path());

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded to Downloads folder.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      } else {
        // On iOS, save to DocumentDirectoryPath and offer to share
        downloadDest = `${ReactNativeBlobUtil.fs.dirs.DocumentDir}/${fileName}`;
        const config = {
          fileCache: true,
          path: downloadDest,
        };

        const res = await ReactNativeBlobUtil.config(config).fetch('GET', pdfUrl);
        console.log('PDF downloaded to:', res.path());

        // Share the file on iOS to allow saving to Files app
        await Share.share({
          url: `file://${res.path()}`,
          message: 'Here is your downloaded PDF.',
        });

        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'PDF downloaded and ready to share.',
          position: 'bottom',
          visibilityTime: 3000,
        });
      }
    } catch (error) {
      console.error('Error downloading PDF:', error);
      let errorMessage = 'Failed to download PDF.';
      if (error.message.includes('permission denied')) {
        errorMessage = 'Storage permission denied. Please grant permission to download.';
      } else if (error.message.includes('Network')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else {
        errorMessage = `Download error: ${error.message}`;
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 3000,
      });
    }
  };

  return (
    <View style={styles.container}>
      {loading ? (
        <Modal visible={loading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.syncModalTextadd}>Downloading PDF...</Text>
              </View>
            </View>
          </View>
        </Modal>
      ) : localPdfPath ? (
        <>
          <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
            <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
            <View style={styles.headerContent}>
              <View style={{ width: 40, paddingLeft: width * 0.04 }}>
                <TouchableOpacity style={styles.backButton} onPress={() => navigation.navigate('HSEInspectionHome')}>
                  <MaterialIcons name="arrow-back" size={width * 0.07} color="white" />
                </TouchableOpacity>
              </View>

              <Text style={styles.headerTitle}>PDF Viewer</Text>

              <View style={{ width: 40, paddingRight: width * 0.04 }} />
            </View>
          </LinearGradient>
          <Pdf
            source={{ uri: Platform.OS === 'android' ? `file://${localPdfPath}` : localPdfPath, cache: false }}
            style={styles.pdf}
            onError={(error) => {
              console.error('PDF rendering error:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to load PDF.',
                position: 'bottom',
                visibilityTime: 3000,
              });
              navigation.goBack();
            }}
            onLoadComplete={(numberOfPages, filePath) => {
              console.log(`PDF loaded with ${numberOfPages} pages at ${filePath}`);
            }}
          />
          <View style={styles.buttonContainer}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.downloadButtonGradient}>
              <TouchableOpacity style={styles.downloadButton} onPress={handleDownloadPDF}>
                <MaterialIcons name="download" size={width * 0.05} color="#fff" />
                <Text style={styles.downloadButtonText}>Download PDF</Text>
              </TouchableOpacity>
            </LinearGradient>
          </View>
        </>
      ) : (
        <View style={styles.errorContainer}>
          <Text style={styles.errorText}>Invalid or unavailable PDF URL</Text>
          <TouchableOpacity onPress={() => navigation.navigate('HSEInspectionHome')}>
            <Text style={styles.closeText}>Go Back</Text>
          </TouchableOpacity>
        </View>
      )}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 0,
    paddingTop: Platform.OS === 'ios' ? width * 0.06 : width * 0.04,
    height: '100%',
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    fontSize: scale(20),
    fontWeight: 'bold',
    color: 'white',
  },
  headerActions: {
    width: 40,
  },
  pdf: {
    flex: 1,
    width: width,
    height: height - (width * 0.2 + (StatusBar.currentHeight || width * 0.1)),
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '5%',
    width: '100%',
    alignItems: 'center',
  },
  downloadButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  downloadButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  downloadButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  errorContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: width * 0.05,
  },
  errorText: {
    fontSize: width * 0.045,
    color: '#FF0000',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  closeText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});


export default HSEViewPDF