// MongoDB API Configuration
// Centralized configuration for all MongoDB backend services

const MONGODB_CONFIG = {
  // Base URLs for all MongoDB services
  BASE_URL: 'https://api.titandrillingzm.com',
  
  // Individual service ports
  PORTS: {
    USERS: 5003,
    EQUIPMENT: 5001,
    DOCUMENTS: 5002,
  },
  
  // Full API endpoints
  APIS: {
    USERS: 'https://api.titandrillingzm.com:6003',
    EQUIPMENT: 'https://api.titandrillingzm.com:6001',
    DOCUMENTS: 'https://api.titandrillingzm.com:6002',
  },
  
  // API Routes
  ROUTES: {
    // Users API Routes
    USERS: {
      GET_ALL: '/users/all',
      GET_FAST: '/users/team-management', // Fast endpoint for team management
      GET_PAGINATED: '/users/paginated',
      GET_OVERVIEW: '/users',
      GET_BY_ID: (userId) => `/users/${userId}`,
      UPLOAD: '/users/upload',
      BULK_UPLOAD: '/users/bulk-upload',
      UPDATE: (userId) => `/users/${userId}`,
      DELETE: (userId) => `/users/${userId}`,
      CLEANUP: '/users/cleanup',
    },
    
    // Equipment API Routes
    EQUIPMENT: {
      GET_ALL: '/equipment',
      GET_BY_ID: (equipmentId) => `/equipment/${equipmentId}`,
      CREATE: '/equipment',
      UPDATE: (equipmentId) => `/equipment/${equipmentId}`,
      DELETE: (equipmentId) => `/equipment/${equipmentId}`,
    },
    
    // Documents API Routes
    DOCUMENTS: {
      GET_ALL: '/GlobalDocumentData',
      GET_OVERVIEW: '/documents',
      GET_USER_DOCS: (userId) => `/Documents/${userId}`,
      GET_BY_SUBFOLDER: (userId, docType, subfolder) => `/Documents/${userId}/${docType}/${subfolder}`,
      UPLOAD: '/Documents/upload',
      SERVE_FILE: (docId) => `/image/${docId}`,
      DELETE: (userId, docType, subfolder, docId) => `/Documents/${userId}/${docType}/${subfolder}/${docId}`,
      ADMIN_DOCUMENTS: '/AdminDocuments',
    },
    // Team Employee Documents API Routes
    TEAM_EMPLOYEE_DOCS: {
      UPLOAD: '/team-employee/upload',
      GET_USER_DOCS: (userId) => `/team-employee/${userId}`,
      SERVE_FILE: (docId) => `/image/${docId}`,
    },
  },
  
  // Request timeout in milliseconds - bumped to 10 seconds to avoid premature aborts on cellular/VPN
  TIMEOUT: 10000,
  
  // Retry configuration
  RETRY: {
    MAX_ATTEMPTS: 2,
    DELAY: 1000,
  },
};

// Helper function to build full URL
export const buildURL = (service, route, params = {}) => {
  const baseURL = MONGODB_CONFIG.APIS[service];
  if (!baseURL) {
    console.error(`❌ Invalid service: ${service}`);
    return null;
  }
  
  let url = baseURL + route;
  
  // Add query parameters if provided
  if (Object.keys(params).length > 0) {
    const queryString = Object.entries(params)
      .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
      .join('&');
    url += `?${queryString}`;
  }
  
  return url;
};

// Helper function to make API calls with retry logic
export const apiCall = async (url, options = {}, retryCount = 0) => {
  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), MONGODB_CONFIG.TIMEOUT);

    const response = await fetch(url, {
      ...options,
      signal: controller.signal,
      headers: {
        'Content-Type': 'application/json',
        ...options.headers,
      },
    });

    clearTimeout(timeoutId);

    const responseData = await response.json().catch(() => null);

    if (!response.ok) {
      const error = new Error(
        responseData?.error ||
        responseData?.message ||
        `HTTP ${response.status}: ${response.statusText}`
      );
      error.status = response.status;
      error.details = responseData;
      throw error;
    }

    return responseData ?? {};
  } catch (error) {
    // Retry logic
    if (retryCount < MONGODB_CONFIG.RETRY.MAX_ATTEMPTS) {
      await new Promise(resolve => setTimeout(resolve, MONGODB_CONFIG.RETRY.DELAY));
      return apiCall(url, options, retryCount + 1);
    }
    
    throw error;
  }
};

// Export configuration
export default MONGODB_CONFIG;

// Export individual APIs for convenience
export const USERS_API = MONGODB_CONFIG.APIS.USERS;
export const EQUIPMENT_API = MONGODB_CONFIG.APIS.EQUIPMENT;
export const DOCUMENTS_API = MONGODB_CONFIG.APIS.DOCUMENTS;

// Export routes
export const USERS_ROUTES = MONGODB_CONFIG.ROUTES.USERS;
export const EQUIPMENT_ROUTES = MONGODB_CONFIG.ROUTES.EQUIPMENT;
export const DOCUMENTS_ROUTES = MONGODB_CONFIG.ROUTES.DOCUMENTS;

// Quick access functions
export const MongoAPI = {
  // Fast Login - search by email or employee ID directly (no timeout issues)
  fastLogin: async (email, password, employeeNumber = null) => {
    const url = buildURL('USERS', '/users/login');
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({ 
        email: email?.trim(), 
        password: password?.trim(),
        employeeNumber: employeeNumber?.trim() 
      }),
    });
  },

  // Search user by email or employee number (without password verification)
  searchUser: async (email, employeeNumber = null) => {
    const params = {};
    if (email) params.email = email;
    if (employeeNumber) params.employeeNumber = employeeNumber;
    
    const url = buildURL('USERS', '/users/search', params);
    return await apiCall(url);
  },

    // Users
    getAllUsers: async () => {
      const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.GET_ALL);
      return await apiCall(url);
    },
    
    getUsersFast: async (page = 1, limit = 50) => {
      const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.GET_FAST, { page, limit });
      return await apiCall(url);
    },
  
  getUsersPaginated: async (page = 1, limit = 20) => {
    const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.GET_PAGINATED, { page, limit });
    return await apiCall(url);
  },
  
  getUserById: async (userId) => {
    const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.GET_BY_ID(userId));
    return await apiCall(url);
  },
  
  updateUser: async (userId, userData) => {
    const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.UPDATE(userId));
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(userData),
    });
  },
  
  createUser: async (userId, userData) => {
    const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.UPLOAD);
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({ userId, userData }),
    });
  },
  
  deleteUser: async (userId) => {
    const url = buildURL('USERS', MONGODB_CONFIG.ROUTES.USERS.DELETE(userId));
    return await apiCall(url, {
      method: 'DELETE',
    });
  },
  
  // Equipment
  getAllEquipment: async () => {
    const url = buildURL('EQUIPMENT', MONGODB_CONFIG.ROUTES.EQUIPMENT.GET_ALL);
    return await apiCall(url);
  },
  
  // Documents
  getAllDocuments: async () => {
    const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.GET_ALL);
    return await apiCall(url);
  },
  
  getDocumentsOverview: async () => {
    const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.GET_OVERVIEW);
    return await apiCall(url);
  },
  
  getUserDocuments: async (userId) => {
    const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.GET_USER_DOCS(userId));
    const response = await apiCall(url);
    
    // Handle different response formats
    if (Array.isArray(response)) {
      return { success: true, data: response };
    } else if (response && response.data) {
      return response;
    } else {
      return { success: true, data: [] };
    }
  },
  
  uploadDocument: async (formData) => {
    const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.UPLOAD);
    return await apiCall(url, {
      method: 'POST',
      headers: {
        // Don't set Content-Type for FormData - browser will set it with boundary
      },
      body: formData,
    });
  },
  
    deleteDocument: async (userId, docType, subfolder, docId) => {
      const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.DELETE(userId, docType, subfolder, docId));
      return await apiCall(url, {
        method: 'DELETE',
      });
    },

  getAdminDocuments: async () => {
    const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.DOCUMENTS.ADMIN_DOCUMENTS);
    return await apiCall(url);
  },

    // Team Employee Documents
    uploadTeamEmployeeDocument: async (formData) => {
      const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.TEAM_EMPLOYEE_DOCS.UPLOAD);
      return await apiCall(url, {
        method: 'POST',
        body: formData,
      });
    },

    getTeamEmployeeDocuments: async (userId) => {
      const url = buildURL('DOCUMENTS', MONGODB_CONFIG.ROUTES.TEAM_EMPLOYEE_DOCS.GET_USER_DOCS(userId));
      return await apiCall(url);
    },

  // FCM Token Management
  updateFCMToken: async (userId, fcmToken, deviceId, platform) => {
    const url = `${MONGODB_CONFIG.APIS.USERS}/users/${userId}/fcm-token`;
    console.log('🔔 Updating FCM token in MongoDB:', { userId, deviceId, platform });
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({ fcmToken, deviceId, platform }),
    });
  },

  getFCMTokens: async (userId) => {
    const url = `${MONGODB_CONFIG.APIS.USERS}/users/${userId}/fcm-tokens`;
    return await apiCall(url);
  },
};

console.log('✅ MongoDB Config loaded successfully');
console.log('📡 Users API:', MONGODB_CONFIG.APIS.USERS);
console.log('📡 Equipment API:', MONGODB_CONFIG.APIS.EQUIPMENT);
console.log('📡 Documents API:', MONGODB_CONFIG.APIS.DOCUMENTS);
