// MongoDB Configuration for Equipment
const EQUIPMENT_API_BASE_URL = 'https://api.titandrillingzm.com:6001';

// Helper function to build URLs with query parameters
const buildURL = (endpoint, params = {}) => {
  const url = new URL(`${EQUIPMENT_API_BASE_URL}${endpoint}`);
  Object.keys(params).forEach(key => {
    if (params[key] !== null && params[key] !== undefined) {
      url.searchParams.append(key, params[key]);
    }
  });
  return url.toString();
};

// API call helper
const apiCall = async (url, options = {}) => {
  try {
    const response = await fetch(url, {
      headers: {
        'Content-Type': 'application/json',
        ...options.headers,
      },
      ...options,
    });

    const data = await response.json();
    
    if (!response.ok) {
      throw new Error(data.error || `HTTP ${response.status}`);
    }
    
    return data;
  } catch (error) {
    console.error('Equipment API call failed:', error);
    throw error;
  }
};

// Equipment API functions
const EquipmentAPI = {
  // Get all equipment
  getAllEquipment: async () => {
    const url = buildURL('/EquipmentsDataJSON/GlobalEquipmentJsonData');
    return await apiCall(url);
  },

  // Get equipment by ID
  getEquipmentById: async (id) => {
    const url = buildURL(`/equipment/${id}`);
    return await apiCall(url);
  },

  // Add single equipment
  addEquipment: async (equipmentData) => {
    const url = buildURL('/add');
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify(equipmentData)
    });
  },

  // Update equipment
  updateEquipment: async (id, equipmentData) => {
    const url = buildURL(`/equipment/${id}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(equipmentData)
    });
  },

  // Delete equipment
  deleteEquipment: async (id) => {
    const url = buildURL(`/equipment/${id}`);
    return await apiCall(url, {
      method: 'DELETE'
    });
  },

  // Bulk upload equipment
  bulkUpload: async (equipmentData) => {
    const url = buildURL('/bulk-upload');
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({ equipmentData })
    });
  },

  // Get equipment count
  getCount: async () => {
    const url = buildURL('/count');
    return await apiCall(url);
  },

  // Clear all equipment
  clearAll: async () => {
    const url = buildURL('/clear-all');
    return await apiCall(url, {
      method: 'DELETE'
    });
  }
};

export default EquipmentAPI;

