const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const app = express();
const PORT = process.env.DVIR_PORT || 5006; // Different port for DVIR operations
const mongoURI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@api.titandrillingzm.com:27017/titan_drilling?authSource=admin';

// Configure file upload storage
// Multer configuration for file uploads - Same pattern as Inspection Handler
const dvirStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    // Store in temp directory first, then move to user-specific folder
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    
    // Ensure temp directory exists
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 Created temp directory: ${tempDir}`);
    }
    
    console.log(`📁 Storing file temporarily in: ${tempDir}`);
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    // Generate unique filename with timestamp and original extension
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});

const fileUpload = multer({ 
  storage: dvirStorage,
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB limit
  },
  fileFilter: function (req, file, cb) {
    // Only allow PDF files
    if (file.mimetype === 'application/pdf') {
      cb(null, true);
    } else {
      cb(new Error('Only PDF files are allowed'), false);
    }
  }
});

// Middleware
app.use(cors({
  origin: '*',
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Request logging middleware
app.use((req, res, next) => {
  console.log(`📡 ${req.method} ${req.url} - ${new Date().toISOString()}`);
  next();
});

// Connect to MongoDB with better error handling
const connectDB = async () => {
  try {
    await mongoose.connect(mongoURI, { 
      serverSelectionTimeoutMS: 10000,
      maxPoolSize: 10,
      socketTimeoutMS: 45000,
      bufferCommands: false,
      bufferMaxEntries: 0
    });
    console.log('✅ MongoDB connected successfully for DVIR Handler!');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    console.error('⚠️ Will retry connection...');
    // Don't exit immediately, let it retry
    setTimeout(() => {
      connectDB();
    }, 5000);
  }
};

connectDB();

// Handle MongoDB connection events
mongoose.connection.on('error', (err) => {
  console.error('❌ MongoDB connection error:', err);
});

mongoose.connection.on('disconnected', () => {
  console.log('⚠️ MongoDB disconnected');
});

mongoose.connection.on('connected', () => {
  console.log('✅ MongoDB connected');
});

// Define DVIR Reports Schema
const DVIRReportSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  reportId: { type: String, required: true },
  dvirNumber: String,
  userUid: { type: String, required: true },
  step1: {
    requestNumber: String,
    country: String,
    project: String,
    inspector: String,
    selectedEquipment: Object,
    meterReading: String,
    currentMeterReading: String,
    inspectionTypes: Object
  },
  step2: {
    items: [{
      section: String,
      label: String,
      status: String,
      note: String
    }],
    checklistId: String,
    timestamp: Number
  },
  step3: {
    faultyItems: Number,
    overallCondition: String,
    equipmentStatus: String,
    priority: String,
    safeToUse: String,
    additionalNote: String,
    signatureStatus: Boolean,
    signatureData: String,
    signatureDownloadUrl: String,
    pdfDownloadUrl: String,
    pdfLocalPath: String, // Local file path on server
    createdAt: { type: Date, default: Date.now },
    timestamp: Number,
    dvirNumber: String
  },
  correctiveAction: Object,
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now }
}, {
  strict: false,
  collection: 'DVIRReports'
});

// Define models with existence check to prevent overwrite errors
const DVIRReport = mongoose.models.DVIRReport || mongoose.model('DVIRReport', DVIRReportSchema);

// Define Corrective Action Schema - Fresh definition to avoid conflicts
const CorrectiveActionSchema = new mongoose.Schema({
  reportId: { 
    type: String, 
    required: true, 
    unique: true,
    index: true 
  },
  userUid: { 
    type: String, 
    required: true,
    index: true 
  },
  dvirNumber: String,
  linkedDvirReportId: String,
  subItemStatuses: {
    type: mongoose.Schema.Types.Mixed,
    default: {}
  },
  isSatisfactory: Boolean,
  role: String,
  mechanicName: String,
  operatorName: String,
  mechanicSignatureUri: String,
  operatorSignatureUri: String,
  pdfDownloadUrl: String,
  pdfLocalPath: String,
  faultyCount: Number,
  status: { 
    type: String, 
    default: 'step1_completed',
    enum: ['step1_completed', 'completed']
  },
  checklistData: {
    items: [mongoose.Schema.Types.Mixed],
    totalItems: Number,
    checkedItems: Number
  },
  originalReportData: {
    step1: mongoose.Schema.Types.Mixed,
    step2: mongoose.Schema.Types.Mixed,
    step3: mongoose.Schema.Types.Mixed
  },
  createdAt: { 
    type: Date, 
    default: Date.now 
  },
  updatedAt: { 
    type: Date, 
    default: Date.now 
  }
}, {
  strict: false,
  collection: 'DVIRcorrectiveActions',
  timestamps: false // We handle timestamps manually
});

// Clear any existing model to prevent schema conflicts
if (mongoose.models.CorrectiveAction) {
  delete mongoose.models.CorrectiveAction;
  console.log('🗑️ Cleared existing CorrectiveAction model');
}

// Add pre-save middleware to ensure data integrity
CorrectiveActionSchema.pre('save', function(next) {
  // Ensure updatedAt is always set
  this.updatedAt = new Date();
  
  // Ensure required fields are present
  if (!this.reportId) {
    return next(new Error('reportId is required'));
  }
  if (!this.userUid) {
    return next(new Error('userUid is required'));
  }
  
  // Ensure status has a default value
  if (!this.status) {
    this.status = 'step1_completed';
  }
  
  console.log('📝 Pre-save validation passed for CorrectiveAction:', {
    reportId: this.reportId,
    userUid: this.userUid,
    status: this.status,
    faultyCount: this.faultyCount
  });
  
  next();
});

// Create fresh model
const CorrectiveAction = mongoose.model('CorrectiveAction', CorrectiveActionSchema);
console.log('✅ Created fresh CorrectiveAction model for DVIRcorrectiveActions collection');

// Health Check Route
app.get('/', (req, res) => {
  res.json({
    status: 'DVIR Handler Server is running',
    port: PORT,
    collections: ['DVIRReports', 'DVIRcorrectiveActions'],
    mongoose: {
      connected: mongoose.connection.readyState === 1,
      state: mongoose.connection.readyState
    }
  });
});

// Test endpoint for corrective action PDF upload debugging
app.post('/test-corrective-upload', fileUpload.single('pdf'), (req, res) => {
  console.log('🧪 Test upload endpoint called');
  console.log('📤 Request details:', {
    method: req.method,
    headers: req.headers,
    body: req.body,
    file: req.file ? {
      fieldname: req.file.fieldname,
      originalname: req.file.originalname,
      mimetype: req.file.mimetype,
      size: req.file.size,
      path: req.file.path
    } : null
  });
  
  res.json({
    success: true,
    message: 'Test upload received',
    received: {
      hasFile: !!req.file,
      bodyKeys: Object.keys(req.body),
      fileInfo: req.file ? {
        fieldname: req.file.fieldname,
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size
      } : null
    }
  });
});

// Test endpoint to verify corrective action schema
app.get('/test-corrective-schema', async (req, res) => {
  try {
    // Create a test document to verify schema
    const testDoc = new CorrectiveAction({
      reportId: 'test-report-123',
      userUid: 'test-user-123',
      status: 'step1_completed',
      faultyCount: 1
    });
    
    // Validate without saving
    const validationError = testDoc.validateSync();
    if (validationError) {
      return res.status(400).json({
        success: false,
        error: 'Schema validation failed',
        details: validationError.message
      });
    }
    
    res.json({
      success: true,
      message: 'CorrectiveAction schema is valid',
      testData: {
        reportId: testDoc.reportId,
        userUid: testDoc.userUid,
        status: testDoc.status,
        faultyCount: testDoc.faultyCount
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Schema test failed',
      details: error.message
    });
  }
});

// Debug endpoint to check file system
app.get('/debug/files/:userUid', (req, res) => {
  try {
    const { userUid } = req.params;
    const dirPath = path.join('/root/node-mongo-api/uploads/documents/DvirReportsdocx', userUid);
    
    console.log('🔍 Debug request for userUid:', userUid);
    console.log('📁 Checking directory:', dirPath);
    
    const dirExists = fs.existsSync(dirPath);
    let files = [];
    
    if (dirExists) {
      try {
        files = fs.readdirSync(dirPath);
      } catch (err) {
        console.error('Error reading directory:', err);
      }
    }
    
    res.json({
      success: true,
      userUid,
      dirPath,
      dirExists,
      files,
      fileCount: files.length
    });
  } catch (err) {
    console.error('❌ Debug error:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Get DVIR reports endpoint
app.get('/reports', async (req, res) => {
  try {
    console.log('📊 Fetching DVIR reports from MongoDB...');
    console.log('Query params:', req.query);
    
    const { userUid, limit, skip, sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
    
    // Build query
    let query = {};
    if (userUid) {
      query.userUid = userUid;
    }
    
    // Build sort object
    const sort = {};
    sort[sortBy] = sortOrder === 'desc' ? -1 : 1;
    
    // Apply pagination
    let queryBuilder = DVIRReport.find(query).sort(sort);
    
    if (limit) {
      queryBuilder = queryBuilder.limit(parseInt(limit));
    }
    if (skip) {
      queryBuilder = queryBuilder.skip(parseInt(skip));
    }
    
    const reports = await queryBuilder.lean();
    const totalCount = await DVIRReport.countDocuments(query);

    console.log(`✅ Successfully fetched ${reports.length} DVIR reports (total: ${totalCount})`);

    res.json({
      success: true,
      count: reports.length,
      totalCount: totalCount,
      data: reports,
      timestamp: new Date().toISOString()
    });
  } catch (err) {
    console.error('❌ Error fetching DVIR reports:', err);
    res.status(500).json({ 
      success: false,
      error: err.message,
      timestamp: new Date().toISOString()
    });
  }
});

// Get reports count endpoint - MUST come before /reports/:id route
app.get('/reports/count', async (req, res) => {
  try {
    console.log('📊 Getting DVIR reports count...');
    console.log('📊 Query params:', req.query);
    console.log('📊 Request headers:', req.headers);
    
    const { userUid } = req.query;
    
    let query = {};
    if (userUid) {
      query.userUid = userUid;
      console.log('📊 Filtering by userUid:', userUid);
    } else {
      console.log('📊 Getting total count for all users');
    }
    
    console.log('📊 MongoDB query:', JSON.stringify(query));
    
    const count = await DVIRReport.countDocuments(query);
    
    console.log('📊 Count result:', count);
    
    res.json({
      success: true,
      count: count,
      query: query,
      timestamp: new Date().toISOString()
    });
  } catch (err) {
    console.error('❌ Error getting reports count:', err);
    console.error('❌ Error details:', {
      message: err.message,
      name: err.name,
      stack: err.stack
    });
    res.status(500).json({
      success: false,
      error: err.message,
      details: 'Failed to get DVIR reports count'
    });
  }
});

// Get DVIR report by ID
app.get('/reports/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const report = await DVIRReport.findById(id);

    if (!report) {
      return res.status(404).json({ 
        success: false,
        error: 'DVIR Report not found' 
      });
    }

    res.json({
      success: true,
      data: report
    });
  } catch (err) {
    console.error('❌ Error fetching DVIR report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Upload PDF to local storage - Following Inspection Handler pattern
app.post('/reports/upload-pdf', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 PDF Upload Request received:', {
      userUid: req.body.userUid,
      reportId: req.body.reportId,
      hasFile: !!req.file,
      fileInfo: req.file ? {
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null,
      headers: req.headers
    });
    
    const { userUid, reportId } = req.body;
    
    if (!userUid || !reportId) {
      return res.status(400).json({
        success: false,
        error: 'userUid and reportId are required'
      });
    }
    
    if (!req.file) {
      console.log('❌ No PDF file in request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const fileName = `dvir_${userUid}_${Date.now()}.pdf`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'dvirreports', userUid);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    let finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      finalPath = req.file.path;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/reports/download-file/${userUid}/${fileName}`,
      message: 'PDF uploaded successfully'
    });

  } catch (error) {
    console.error('PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload PDF: ' + error.message
    });
  }
});

// Download file from local storage - Following Inspection Handler pattern
app.get('/reports/download-file/:userUid/:fileName', (req, res) => {
  try {
    const { userUid, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'dvirreports', userUid, fileName);
    
    console.log('🔍 Looking for PDF file at:', filePath);
    console.log('🔍 User ID:', userUid);
    console.log('🔍 File name:', fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      console.log('❌ File not found at:', filePath);
      return res.status(404).json({
        success: false,
        error: 'PDF file not found'
      });
    }
    
    console.log('✅ File found at:', filePath);

    // Set appropriate headers
    const ext = path.extname(fileName).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.png') {
      contentType = 'image/png';
    } else if (ext === '.jpg' || ext === '.jpeg') {
      contentType = 'image/jpeg';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// Legacy download endpoint for backward compatibility
app.get('/download/:userUid/:filename', (req, res) => {
  // Redirect to new download-file endpoint
  const { userUid, filename } = req.params;
  res.redirect(`/reports/download-file/${userUid}/${filename}`);
});

// Create new DVIR report (with optional PDF upload)
app.post('/reports', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 Creating new DVIR report...');
    console.log('Request body keys:', Object.keys(req.body));
    console.log('Uploaded file:', req.file ? req.file.filename : 'No file uploaded');
    
    const reportData = req.body;
    
    // Validate required fields
    if (!reportData.reportId && !reportData.id) {
      return res.status(400).json({
        success: false,
        error: 'reportId or id is required'
      });
    }
    
    if (!reportData.userUid) {
      return res.status(400).json({
        success: false,
        error: 'userUid is required'
      });
    }

    // Use reportId or id as the document _id
    const documentId = reportData.reportId || reportData.id;

    // Handle PDF file upload
    let pdfInfo = null;
    if (req.file) {
      const userUid = reportData.userUid;
      const fileName = `dvir_${userUid}_${Date.now()}.pdf`;
      
      // Create user-specific directory
      const userDir = path.join(__dirname, 'uploads', 'documents', 'dvirreports', userUid);
      if (!fs.existsSync(userDir)) {
        fs.mkdirSync(userDir, { recursive: true });
        console.log(`📁 Created user directory: ${userDir}`);
      }
      
      // Move file from temp to user directory
      let finalPath = path.join(userDir, fileName);
      
      try {
        fs.renameSync(req.file.path, finalPath);
        console.log(`📁 File moved to user directory: ${finalPath}`);
      } catch (moveError) {
        console.error('❌ Error moving file to user directory:', moveError);
        // If move fails, use the temp path
        finalPath = req.file.path;
      }
      
      const fileUrl = `https://api.titandrillingzm.com:5006/reports/download-file/${userUid}/${fileName}`;
      pdfInfo = {
        filename: fileName,
        localPath: finalPath,
        downloadUrl: fileUrl
      };
      
      // Add PDF info to step3 if it exists
      if (reportData.step3) {
        reportData.step3.pdfLocalPath = finalPath;
        reportData.step3.pdfDownloadUrl = fileUrl;
      } else if (reportData.step3 === undefined) {
        reportData.step3 = {
          pdfLocalPath: finalPath,
          pdfDownloadUrl: fileUrl
        };
      }
    }
    
    // Check if report already exists
    const existingReport = await DVIRReport.findById(documentId);
    if (existingReport) {
      console.log(`🔄 Updating existing DVIR report: ${documentId}`);
      
      // Update existing report
      const updatedReport = await DVIRReport.findByIdAndUpdate(
        documentId,
        {
          ...reportData,
          updatedAt: new Date(),
          _id: documentId // Ensure _id is preserved
        },
        { new: true, runValidators: false }
      );

      console.log(`✅ DVIR report updated: ${documentId}`);

      return res.json({
        success: true,
        message: 'DVIR report updated successfully',
        isUpdate: true,
        data: updatedReport,
        pdfInfo: pdfInfo
      });
    }

    // Create new report
    const newReport = new DVIRReport({
      _id: documentId,
      reportId: documentId,
      ...reportData,
      createdAt: new Date(),
      updatedAt: new Date()
    });

    const savedReport = await newReport.save();

    console.log(`✅ DVIR report created: ${documentId}`);

    res.json({
      success: true,
      message: 'DVIR report created successfully',
      isUpdate: false,
      data: savedReport,
      pdfInfo: pdfInfo
    });
  } catch (err) {
    console.error('❌ Error creating/updating DVIR report:', err);
    console.error('Error details:', {
      message: err.message,
      name: err.name,
      code: err.code
    });
    
    res.status(500).json({
      success: false,
      error: err.message,
      details: err.code === 11000 ? 'Report ID already exists' : err.message
    });
  }
});

// Update DVIR report
app.put('/reports/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    const result = await DVIRReport.findByIdAndUpdate(
      id,
      { ...updateData, updatedAt: new Date() },
      { new: true, runValidators: false }
    );

    if (!result) {
      return res.status(404).json({ 
        success: false,
        error: 'DVIR Report not found' 
      });
    }

    res.json({
      success: true,
      message: 'DVIR report updated successfully',
      data: result
    });
  } catch (err) {
    console.error('❌ Error updating DVIR report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Delete DVIR report
app.delete('/reports/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const result = await DVIRReport.findByIdAndDelete(id);

    if (!result) {
      return res.status(404).json({ 
        success: false,
        error: 'DVIR Report not found' 
      });
    }

    res.json({
      success: true,
      message: 'DVIR report deleted successfully',
      deletedReport: {
        _id: result._id,
        reportId: result.reportId
      }
    });
  } catch (err) {
    console.error('❌ Error deleting DVIR report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Bulk operations endpoint
app.post('/reports/bulk', async (req, res) => {
  try {
    console.log('📤 Bulk creating/updating DVIR reports...');
    
    const { reports } = req.body;
    
    if (!Array.isArray(reports)) {
      return res.status(400).json({
        success: false,
        error: 'reports must be an array'
      });
    }

    const results = [];
    let successCount = 0;
    let errorCount = 0;

    for (const reportData of reports) {
      try {
        const documentId = reportData.reportId || reportData.id;
        
        if (!documentId || !reportData.userUid) {
          results.push({
            id: documentId || 'unknown',
            success: false,
            error: 'reportId and userUid are required'
          });
          errorCount++;
          continue;
        }

        const existingReport = await DVIRReport.findById(documentId);
        
        let result;
        if (existingReport) {
          result = await DVIRReport.findByIdAndUpdate(
            documentId,
            { ...reportData, updatedAt: new Date() },
            { new: true, runValidators: false }
          );
        } else {
          const newReport = new DVIRReport({
            _id: documentId,
            reportId: documentId,
            ...reportData,
            createdAt: new Date(),
            updatedAt: new Date()
          });
          result = await newReport.save();
        }

        results.push({
          id: documentId,
          success: true,
          isUpdate: !!existingReport,
          data: result
        });
        successCount++;
      } catch (reportError) {
        console.error(`❌ Error processing report ${reportData.reportId || reportData.id}:`, reportError.message);
        results.push({
          id: reportData.reportId || reportData.id || 'unknown',
          success: false,
          error: reportError.message
        });
        errorCount++;
      }
    }

    console.log(`📊 Bulk operation completed: ${successCount} success, ${errorCount} errors`);

    res.json({
      success: true,
      message: `Bulk operation completed: ${successCount} success, ${errorCount} errors`,
      totalReports: reports.length,
      successCount,
      errorCount,
      results
    });
  } catch (err) {
    console.error('❌ Error in bulk operation:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// ========== CORRECTIVE ACTION ENDPOINTS ==========

// Get corrective actions endpoint
app.get('/dvir-corrective-actions', async (req, res) => {
  try {
    console.log('📊 Fetching corrective actions from MongoDB...');
    console.log('Query params:', req.query);
    
    const { userUid, reportId, limit, skip, sortBy = 'createdAt', sortOrder = 'desc' } = req.query;
    
    // Build query
    let query = {};
    if (userUid) {
      query.userUid = userUid;
    }
    if (reportId) {
      query.reportId = reportId;
    }
    
    // Build sort object
    const sort = {};
    sort[sortBy] = sortOrder === 'desc' ? -1 : 1;
    
    // Apply pagination
    let queryBuilder = CorrectiveAction.find(query).sort(sort);
    
    if (limit) {
      queryBuilder = queryBuilder.limit(parseInt(limit));
    }
    if (skip) {
      queryBuilder = queryBuilder.skip(parseInt(skip));
    }
    
    const correctiveActions = await queryBuilder.lean();
    const totalCount = await CorrectiveAction.countDocuments(query);

    console.log(`✅ Successfully fetched ${correctiveActions.length} corrective actions (total: ${totalCount})`);

    res.json({
      success: true,
      count: correctiveActions.length,
      totalCount: totalCount,
      data: correctiveActions,
      timestamp: new Date().toISOString()
    });
  } catch (err) {
    console.error('❌ Error fetching corrective actions:', err);
    res.status(500).json({ 
      success: false,
      error: err.message,
      timestamp: new Date().toISOString()
    });
  }
});

// Get corrective action by ID
app.get('/dvir-corrective-actions/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log('🔍 Fetching corrective action by ID:', id);
    
    const correctiveAction = await CorrectiveAction.findById(id).lean();
    
    if (!correctiveAction) {
      return res.status(404).json({
        success: false,
        error: 'Corrective action not found'
      });
    }

    console.log('✅ Corrective action found:', id);
    res.json({
      success: true,
      data: correctiveAction
    });
  } catch (err) {
    console.error('❌ Error fetching corrective action:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Get corrective action by report ID
app.get('/dvir-corrective-actions/report/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    const { userUid } = req.query;
    console.log('🔍 Fetching corrective action for report ID:', reportId);
    
    let query = { reportId };
    if (userUid) {
      query.userUid = userUid;
    }
    
    const correctiveAction = await CorrectiveAction.findOne(query).lean();
    
    if (!correctiveAction) {
      return res.status(404).json({
        success: false,
        error: 'Corrective action not found for this report'
      });
    }

    console.log('✅ Corrective action found for report:', reportId);
    res.json({
      success: true,
      data: correctiveAction
    });
  } catch (err) {
    console.error('❌ Error fetching corrective action for report:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Create or update corrective action
app.post('/dvir-corrective-actions', async (req, res) => {
  try {
    console.log('📤 Creating/updating corrective action...');
    console.log('Request body keys:', Object.keys(req.body));
    console.log('Full request body:', JSON.stringify(req.body, null, 2));
    
    const correctiveActionData = req.body;
    
    // Log the incoming data to debug any field issues
    console.log('📋 Raw request data analysis:', {
      hasReportId: !!correctiveActionData.reportId,
      hasUserUid: !!correctiveActionData.userUid,
      hasUserId: !!correctiveActionData.userId,
      hasId: !!correctiveActionData._id,
      allKeys: Object.keys(correctiveActionData)
    });
    
    // Validate required fields
    if (!correctiveActionData.reportId) {
      return res.status(400).json({
        success: false,
        error: 'reportId is required'
      });
    }
    
    if (!correctiveActionData.userUid) {
      return res.status(400).json({
        success: false,
        error: 'userUid is required'
      });
    }

    // Use reportId as the document _id for corrective actions  
    const documentId = `ca_${correctiveActionData.reportId}`;
    
    console.log(`🔍 Looking for existing corrective action with ID: ${documentId}`);
    
    // Check if corrective action already exists using findOne instead of findById
    const existingCorrectiveAction = await CorrectiveAction.findOne({ reportId: correctiveActionData.reportId });
    if (existingCorrectiveAction) {
      console.log(`🔄 Updating existing corrective action for reportId: ${correctiveActionData.reportId}`);
      
      // Update existing corrective action - prepare update data without _id
      const updateData = {
        reportId: correctiveActionData.reportId,
        userUid: correctiveActionData.userUid,
        dvirNumber: correctiveActionData.dvirNumber,
        linkedDvirReportId: correctiveActionData.linkedDvirReportId,
        subItemStatuses: correctiveActionData.subItemStatuses,
        faultyCount: correctiveActionData.faultyCount,
        status: correctiveActionData.status || 'step1_completed',
        checklistData: correctiveActionData.checklistData,
        originalReportData: correctiveActionData.originalReportData,
        isSatisfactory: correctiveActionData.isSatisfactory,
        role: correctiveActionData.role,
        mechanicName: correctiveActionData.mechanicName,
        operatorName: correctiveActionData.operatorName,
        mechanicSignatureUri: correctiveActionData.mechanicSignatureUri,
        operatorSignatureUri: correctiveActionData.operatorSignatureUri,
        pdfDownloadUrl: correctiveActionData.pdfDownloadUrl,
        updatedAt: new Date()
      };
      
      console.log(`📝 Updating corrective action with data:`, JSON.stringify(updateData, null, 2));
      
      const updatedCorrectiveAction = await CorrectiveAction.findOneAndUpdate(
        { reportId: correctiveActionData.reportId },
        updateData,
        { new: true, runValidators: false }
      );

      console.log(`✅ Corrective action updated for reportId: ${correctiveActionData.reportId}`);

      return res.json({
        success: true,
        message: 'Corrective action updated successfully',
        isUpdate: true,
        data: updatedCorrectiveAction
      });
    }

    // Create new corrective action
    console.log(`➕ Creating new corrective action for reportId: ${correctiveActionData.reportId}`);
    
    // Prepare the data for creation - ensure all required fields are present
    const actionData = {
      reportId: correctiveActionData.reportId,
      userUid: correctiveActionData.userUid,
      dvirNumber: correctiveActionData.dvirNumber,
      linkedDvirReportId: correctiveActionData.linkedDvirReportId,
      subItemStatuses: correctiveActionData.subItemStatuses,
      faultyCount: correctiveActionData.faultyCount,
      status: correctiveActionData.status || 'step1_completed',
      checklistData: correctiveActionData.checklistData,
      originalReportData: correctiveActionData.originalReportData,
      isSatisfactory: correctiveActionData.isSatisfactory,
      role: correctiveActionData.role,
      mechanicName: correctiveActionData.mechanicName,
      operatorName: correctiveActionData.operatorName,
      mechanicSignatureUri: correctiveActionData.mechanicSignatureUri,
      operatorSignatureUri: correctiveActionData.operatorSignatureUri,
      pdfDownloadUrl: correctiveActionData.pdfDownloadUrl,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    
    console.log(`📝 Corrective action data being created:`, JSON.stringify(actionData, null, 2));
    
    // Validate required fields before creating
    if (!actionData.reportId || !actionData.userUid) {
      console.error('❌ Missing required fields:', {
        reportId: !!actionData.reportId,
        userUid: !!actionData.userUid
      });
      return res.status(400).json({
        success: false,
        error: 'Missing required fields: reportId or userUid'
      });
    }
    
    // Create the corrective action document
    console.log(`🔧 Creating CorrectiveAction model instance...`);
    const newCorrectiveAction = new CorrectiveAction(actionData);

    console.log(`💾 Attempting to save corrective action for reportId: ${correctiveActionData.reportId}`);
    
    try {
      const savedCorrectiveAction = await newCorrectiveAction.save();
      console.log(`✅ New corrective action created with ID: ${savedCorrectiveAction._id}`);
      
      // Continue with updating the main DVIR report...
      try {
        await DVIRReport.findByIdAndUpdate(
          correctiveActionData.reportId,
          {
            $set: {
              'correctiveAction.hasCorrectiveAction': true,
              'correctiveAction.correctiveActionId': savedCorrectiveAction._id,
              'correctiveAction.status': correctiveActionData.status || 'step1_completed',
              'correctiveAction.updatedAt': new Date().toISOString(),
              'correctiveAction.faultyCount': correctiveActionData.faultyCount
            }
          },
          { upsert: true }
        );
        console.log(`✅ Updated main DVIR report with corrective action reference: ${correctiveActionData.reportId}`);
      } catch (updateError) {
        console.warn('⚠️ Failed to update main DVIR report with corrective action reference:', updateError.message);
      }

      res.status(201).json({
        success: true,
        message: 'Corrective action created successfully',
        isUpdate: false,
        data: savedCorrectiveAction
      });
      
    } catch (saveError) {
      console.error('❌ Error saving corrective action:', saveError);
      console.error('❌ Save error details:', {
        name: saveError.name,
        message: saveError.message,
        code: saveError.code,
        errors: saveError.errors
      });
      
      // Check if it's a validation error
      if (saveError.name === 'ValidationError') {
        const validationErrors = Object.keys(saveError.errors).map(field => ({
          field,
          message: saveError.errors[field].message
        }));
        console.error('❌ Validation errors:', validationErrors);
      }
      
      return res.status(500).json({
        success: false,
        error: saveError.message,
        details: saveError.message,
        validationErrors: saveError.errors ? Object.keys(saveError.errors) : undefined
      });
    }
    
  } catch (err) {
    console.error('❌ Error creating/updating corrective action:', err);
    console.error('❌ Error stack:', err.stack);
    console.error('❌ Error details:', {
      name: err.name,
      message: err.message,
      code: err.code
    });
    res.status(500).json({
      success: false,
      error: err.message,
      details: err.message,
      stack: process.env.NODE_ENV === 'development' ? err.stack : undefined
    });
  }
});

// Upload PDF for corrective action
app.post('/dvir-corrective-actions/upload-pdf', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 Corrective Action PDF Upload Request received:', {
      userUid: req.body.userUid,
      reportId: req.body.reportId,
      hasFile: !!req.file,
      fileInfo: req.file ? {
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null,
      bodyKeys: Object.keys(req.body),
      headers: req.headers
    });
    
    const { userUid, reportId } = req.body;
    
    if (!userUid || !reportId) {
      return res.status(400).json({
        success: false,
        error: 'userUid and reportId are required'
      });
    }
    
    if (!req.file) {
      console.log('❌ No PDF file in request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const fileName = `corrective-action_${userUid}_${reportId}_${Date.now()}.pdf`;
    
    // Create user-specific directory for corrective actions - use same pattern as DVIR reports
    const userDir = path.join(__dirname, 'uploads', 'documents', 'dvirreports', userUid);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created corrective action user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    let finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 Corrective action PDF moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving corrective action PDF:', moveError);
      // If move fails, use the temp path
      finalPath = req.file.path;
    }
    
    const downloadUrl = `https://api.titandrillingzm.com:5006/dvir-corrective-actions/download-file/${userUid}/${fileName}`;
    
    console.log('✅ Corrective action PDF uploaded successfully:', {
      fileName,
      finalPath,
      downloadUrl
    });

    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: downloadUrl,
      message: 'Corrective action PDF uploaded successfully'
    });

  } catch (error) {
    console.error('❌ Corrective action PDF upload error:', error);
    console.error('❌ Error details:', {
      name: error.name,
      message: error.message,
      stack: error.stack
    });
    
    // Clean up temp file if it exists
    if (req.file && fs.existsSync(req.file.path)) {
      try {
        fs.unlinkSync(req.file.path);
        console.log('🗑️ Cleaned up temp file:', req.file.path);
      } catch (cleanupError) {
        console.error('❌ Failed to clean up temp file:', cleanupError);
      }
    }
    
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action PDF: ' + error.message,
      details: error.message
    });
  }
});

// Download corrective action PDF file
app.get('/dvir-corrective-actions/download-file/:userUid/:fileName', (req, res) => {
  try {
    const { userUid, fileName } = req.params;
    // Use same path as DVIR reports since we're storing in the same directory
    const filePath = path.join(__dirname, 'uploads', 'documents', 'dvirreports', userUid, fileName);
    
    console.log(`📥 Corrective action PDF download requested: ${filePath}`);
    
    if (!fs.existsSync(filePath)) {
      console.log(`❌ Corrective action PDF file not found: ${filePath}`);
      return res.status(404).json({
        success: false,
        error: 'Corrective action PDF file not found'
      });
    }
    
    // Set appropriate headers for PDF
    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`
    });
    
    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);
    
    fileStream.on('error', (error) => {
      console.error(`❌ Error sending corrective action PDF file: ${error.message}`);
      res.status(500).json({
        success: false,
        error: 'Failed to download corrective action PDF file'
      });
    });
    
  } catch (error) {
    console.error('Corrective action PDF download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download corrective action PDF: ' + error.message
    });
  }
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('❌ Server error:', err);
  res.status(500).json({
    success: false,
    error: 'Internal server error',
    message: err.message
  });
});

// Handle non-existent routes
app.use('*', (req, res) => {
  res.status(404).json({
    success: false,
    error: 'Route not found',
    path: req.originalUrl,
    method: req.method
  });
});

// Start Server with better error handling
const startServer = () => {
  try {
    const server = app.listen(PORT, '0.0.0.0', (err) => {
      if (err) {
        console.error(`❌ Failed to start server on port ${PORT}:`, err);
        process.exit(1);
      }
      
      console.log(`🚀 DVIR Handler Server running on port ${PORT}`);
      console.log(`📡 Server URL: https://api.titandrillingzm.com:${PORT}`);
      console.log(`🔗 Health check: https://api.titandrillingzm.com:${PORT}/`);
      console.log(`📋 DVIR Reports collection: DVIRReports`);
      console.log(`📋 Corrective Actions collection: DVIRcorrectiveActions`);
      console.log(`📁 PDF Storage: /root/node-mongo-api/uploads/documents/DvirReportsdocx/userUid/`);
      console.log(`📊 Available endpoints:`);
      console.log(`   GET  /reports - Get all reports`);
      console.log(`   GET  /reports/count - Get reports count`);
      console.log(`   GET  /reports/:id - Get report by ID`);
      console.log(`   POST /reports - Create/update report (with optional PDF upload)`);
      console.log(`   POST /reports/upload-pdf - Upload PDF file`);
      console.log(`   GET  /reports/download-file/:userUid/:fileName - Download PDF file (new)`);
      console.log(`   GET  /download/:userUid/:filename - Download PDF file (legacy, redirects)`);
      console.log(`   PUT  /reports/:id - Update report`);
      console.log(`   DELETE /reports/:id - Delete report`);
      console.log(`   POST /reports/bulk - Bulk operations`);
  console.log(`   GET  /dvir-corrective-actions - Get all corrective actions`);
  console.log(`   GET  /dvir-corrective-actions/:id - Get corrective action by ID`);
  console.log(`   GET  /dvir-corrective-actions/report/:reportId - Get corrective action by report ID`);
  console.log(`   POST /dvir-corrective-actions - Create/update corrective action`);
  console.log(`   POST /dvir-corrective-actions/upload-pdf - Upload corrective action PDF`);
  console.log(`   GET  /dvir-corrective-actions/download-file/:userUid/:fileName - Download corrective action PDF`);
  console.log(`   📊 Collection: DVIRcorrectiveActions`);
    });

    return server;
  } catch (error) {
    console.error('❌ Error starting server:', error);
    process.exit(1);
  }
};

// Global server variable to handle graceful shutdown
let server = null;

// Only start server after MongoDB connection is established
mongoose.connection.once('open', () => {
  console.log('✅ MongoDB connection established, starting server...');
  server = startServer();
  
  // Handle server errors
  server.on('error', (err) => {
    if (err.code === 'EADDRINUSE') {
      console.error(`❌ Port ${PORT} is already in use`);
      process.exit(1);
    } else {
      console.error('❌ Server error:', err);
    }
  });
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('🛑 SIGTERM received, shutting down gracefully');
  if (server) {
    server.close(() => {
      console.log('✅ Process terminated');
      mongoose.connection.close();
    });
  } else {
    mongoose.connection.close();
  }
});

process.on('SIGINT', () => {
  console.log('🛑 SIGINT received, shutting down gracefully');
  if (server) {
    server.close(() => {
      console.log('✅ Process terminated');
      mongoose.connection.close();
    });
  } else {
    mongoose.connection.close();
  }
});
