const express = require('express');
const cors = require('cors');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { mongoose, connectDB } = require('./dbConnection');

const app = express();
const PORT = process.env.RISK_ASSESSMENT_PORT || 5031;

app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

console.log('✅ [Risk Assessment] Using shared MongoDB connection');

// Ensure MongoDB connection is established when this service starts standalone
(async () => {
  try {
    await connectDB();
    console.log('✅ [Risk Assessment] MongoDB connected');
  } catch (connectionError) {
    console.error('❌ [Risk Assessment] MongoDB connection failed:', connectionError);
  }
})();

const baseSchemaOptions = {
  strict: false,
};

const riskAssessmentSchema =
  mongoose.models.RiskAssessment?.schema ||
  new mongoose.Schema(
    {
      _id: { type: String, required: true },
      inspectionId: { type: String, required: true },
      adminId: { type: String, default: null },
      sections: mongoose.Schema.Types.Mixed,
      metadata: mongoose.Schema.Types.Mixed,
      createdAt: { type: Date, default: Date.now },
      updatedAt: { type: Date, default: Date.now },
      createdBy: String,
      lastModifiedBy: String,
    },
    baseSchemaOptions
  );

riskAssessmentSchema.index({ inspectionId: 1 }, { unique: true });

const RiskAssessment =
  mongoose.models.RiskAssessment ||
  mongoose.model('RiskAssessment', riskAssessmentSchema, 'riskassessments');

const riskManagementSchema =
  mongoose.models.RiskManagementRecord?.schema ||
  new mongoose.Schema({
    _id: { type: String, required: true },
    adminId: { type: String, required: true },
    inspectionId: { type: String, required: true },
    createdBy: String,
    lastModifiedBy: String,
    linkedInspection: {
      requestNumber: String,
      inspectionId: String,
      jobCardRequestNumber: String,
      jobCardId: String,
      inspector: String,
    },
    step1: {
      selectedHazardIds: [String],
      hazardRiskSelections: mongoose.Schema.Types.Mixed,
      updatedAt: Date,
    },
    step2: {
      rows: [
        {
          step: String,
          hazard: String,
          assess: String,
          control: String,
        },
      ],
      guidanceNotes: [String],
      updatedAt: Date,
    },
    step3: {
      form: mongoose.Schema.Types.Mixed,
      pdfDownloadUrl: String,
      pdfFileName: String,
      generatedAt: Date,
      updatedAt: Date,
    },
    pdfDownloadUrl: String,
    pdfFileName: String,
    createdAt: { type: Date, default: Date.now },
    updatedAt: { type: Date, default: Date.now },
  });

riskManagementSchema.index({ adminId: 1, inspectionId: 1 }, { unique: true });
riskManagementSchema.index({ inspectionId: 1 });

const RiskManagementRecord =
  mongoose.models.RiskManagementRecord ||
  mongoose.model('RiskManagementRecord', riskManagementSchema, 'riskmanagementrecords');

const buildAssessmentDocumentId = (inspectionId) => `risk_assessment_${inspectionId}`;
const getRiskManagementDocumentId = (adminId, inspectionId) => `risk_${adminId}_${inspectionId}`;

const riskStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 [Risk Assessment] Created temp directory: ${tempDir}`);
    }
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
    const ext = path.extname(file.originalname) || '.pdf';
    cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
  },
});

const fileUpload = multer({
  storage: riskStorage,
  limits: { fileSize: 50 * 1024 * 1024 },
});

const upsertRiskManagementRecord = async ({ adminId, inspectionId, setData, createdBy }) => {
  const now = new Date();
  const documentId = getRiskManagementDocumentId(adminId, inspectionId);

  const updateSet = {
    ...setData,
    updatedAt: now,
  };

  const setOnInsert = {
    _id: documentId,
    adminId,
    inspectionId,
    createdAt: now,
  };

  if (createdBy) {
    setOnInsert.createdBy = createdBy;
  }

  console.log('🛠️ [Risk Management] Upsert payload:', {
    documentId,
    updateSet,
    setOnInsert,
  });

  const record = await RiskManagementRecord.findOneAndUpdate(
    { _id: documentId },
    {
      $set: updateSet,
      $setOnInsert: setOnInsert,
    },
    {
      upsert: true,
      new: true,
      runValidators: true,
    }
  ).select('-__v');

  return record;
};

app.get('/risk-assessments', async (req, res) => {
  try {
    const { adminId, page = 1, limit = 50 } = req.query;
    const query = {};

    if (adminId) {
      query.adminId = adminId;
    }

    const safeLimit = Math.min(parseInt(limit, 10) || 50, 200);
    const skip = (parseInt(page, 10) - 1) * safeLimit;

    const [items, total] = await Promise.all([
      RiskAssessment.find(query)
        .sort({ updatedAt: -1 })
        .skip(skip)
        .limit(safeLimit)
        .select('-__v'),
      RiskAssessment.countDocuments(query),
    ]);

    res.json({
      success: true,
      data: items,
      pagination: {
        total,
        page: parseInt(page, 10),
        limit: safeLimit,
        pages: Math.ceil(total / safeLimit),
      },
    });
  } catch (error) {
    console.error('❌ Error listing risk assessments:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch risk assessments',
    });
  }
});

app.get('/risk-assessments/:inspectionId', async (req, res) => {
  try {
    const { inspectionId } = req.params;
    const documentId = buildAssessmentDocumentId(inspectionId);

    const assessment = await RiskAssessment.findOne({ _id: documentId }).select(
      '-__v'
    );

    if (!assessment) {
      return res.status(404).json({
        success: false,
        error: 'Risk assessment not found for this inspection',
      });
    }

    res.json({
      success: true,
      data: assessment,
    });
  } catch (error) {
    console.error('❌ Error fetching risk assessment:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch risk assessment',
    });
  }
});

app.put('/risk-assessments/:inspectionId', async (req, res) => {
  try {
    const { inspectionId } = req.params;
    const payload = req.body || {};

    if (!payload.sections && !payload.metadata) {
      return res.status(400).json({
        success: false,
        error: 'sections or metadata payload is required',
      });
    }

    const now = new Date();
    const documentId = buildAssessmentDocumentId(inspectionId);

    const updateSet = {
      inspectionId,
      adminId: payload.adminId || null,
      sections: payload.sections || {},
      metadata: payload.metadata || {},
      updatedAt: now,
      lastModifiedBy: payload.updatedBy || payload.userId || payload.adminId || null,
    };

    if (payload.summary) {
      updateSet.summary = payload.summary;
    }

    if (payload.actions) {
      updateSet.actions = payload.actions;
    }

    const setOnInsert = {
      _id: documentId,
      inspectionId,
      adminId: payload.adminId || null,
      createdAt: now,
      createdBy: payload.createdBy || payload.userId || payload.adminId || null,
    };

    const assessment = await RiskAssessment.findOneAndUpdate(
      { _id: documentId },
      {
        $set: updateSet,
        $setOnInsert: setOnInsert,
      },
      {
        upsert: true,
        new: true,
        runValidators: true,
      }
    ).select('-__v');

    res.json({
      success: true,
      data: assessment,
      message: 'Risk assessment saved successfully',
    });
  } catch (error) {
    console.error('❌ Error saving risk assessment:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk assessment',
    });
  }
});

app.delete('/risk-assessments/:inspectionId', async (req, res) => {
  try {
    const { inspectionId } = req.params;
    const documentId = buildAssessmentDocumentId(inspectionId);

    const result = await RiskAssessment.findOneAndDelete({ _id: documentId });

    if (!result) {
      return res.status(404).json({
        success: false,
        error: 'Risk assessment not found',
      });
    }

    res.json({
      success: true,
      message: 'Risk assessment deleted successfully',
    });
  } catch (error) {
    console.error('❌ Error deleting risk assessment:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to delete risk assessment',
    });
  }
});

app.get('/risk-management/:adminId/:inspectionId', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;

    const record = await RiskManagementRecord.findOne({ adminId, inspectionId }).select('-__v');

    if (!record) {
      return res.json({
        success: true,
        data: null,
        message: 'Risk management record not found',
        status: 'not_found',
      });
    }

    res.json({
      success: true,
      data: record,
    });
  } catch (error) {
    console.error('❌ [Risk Management] Error fetching record:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch risk management record',
    });
  }
});

app.put('/risk-management/:adminId/:inspectionId/step1', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const step1Data = {
      selectedHazardIds: Array.isArray(payload.selectedHazardIds) ? payload.selectedHazardIds : [],
      hazardRiskSelections: payload.hazardRiskSelections || {},
      updatedAt: payload.updatedAt ? new Date(payload.updatedAt) : now,
    };

    const setData = {
      step1: step1Data,
    };

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 1 saved',
    });
  } catch (error) {
    console.error('❌ [Risk Management] Error saving step1:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step1',
      details: error?.message || 'Unknown error',
    });
  }
});

app.put('/risk-management/:adminId/:inspectionId/step2', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const rows = Array.isArray(payload.rows) ? payload.rows : [];

    const step2Data = {
      rows,
      guidanceNotes: Array.isArray(payload.guidanceNotes) ? payload.guidanceNotes : undefined,
      updatedAt: payload.updatedAt ? new Date(payload.updatedAt) : now,
    };

    if (!step2Data.guidanceNotes) {
      delete step2Data.guidanceNotes;
    }

    const setData = {
      step2: step2Data,
    };

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 2 saved',
    });
  } catch (error) {
    console.error('❌ [Risk Management] Error saving step2:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step2',
      details: error?.message || 'Unknown error',
    });
  }
});

app.put('/risk-management/:adminId/:inspectionId/step3', async (req, res) => {
  try {
    const { adminId, inspectionId } = req.params;
    const payload = req.body || {};
    const now = new Date();

    const step3Data = {
      form: payload.form || {},
      pdfDownloadUrl: payload.pdfDownloadUrl || null,
      pdfFileName: payload.pdfFileName || null,
      generatedAt: payload.generatedAt ? new Date(payload.generatedAt) : now,
      updatedAt: now,
    };

    const setData = {
      step3: step3Data,
    };

    if (payload.pdfDownloadUrl) {
      setData.pdfDownloadUrl = payload.pdfDownloadUrl;
    }
    if (payload.pdfFileName) {
      setData.pdfFileName = payload.pdfFileName;
    }

    if (payload.linkedInspection) {
      setData.linkedInspection = payload.linkedInspection;
    }

    if (payload.userId || payload.updatedBy || payload.lastModifiedBy) {
      setData.lastModifiedBy = payload.updatedBy || payload.lastModifiedBy || payload.userId;
    }

    const record = await upsertRiskManagementRecord({
      adminId,
      inspectionId,
      setData,
      createdBy: payload.createdBy || payload.userId || payload.updatedBy,
    });

    res.json({
      success: true,
      data: record,
      message: 'Risk management step 3 saved',
    });
  } catch (error) {
    console.error('❌ [Risk Management] Error saving step3:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save risk management step3',
      details: error?.message || 'Unknown error',
    });
  }
});

app.post(
  '/risk-management/upload-pdf/:adminId/:inspectionId',
  fileUpload.single('pdf'),
  async (req, res) => {
    try {
      const { adminId, inspectionId } = req.params;
      const { userId } = req.body;

      if (!req.file) {
        return res.status(400).json({
          success: false,
          error: 'No PDF file uploaded',
        });
      }

      const directoryPath = path.join(
        __dirname,
        'uploads',
        'documents',
        'riskmanagement',
        adminId || 'unknown'
      );
      if (!fs.existsSync(directoryPath)) {
        fs.mkdirSync(directoryPath, { recursive: true });
        console.log(`📁 [Risk Management] Created directory: ${directoryPath}`);
      }

      const fileName =
        req.file.originalname && req.file.originalname.endsWith('.pdf')
          ? req.file.originalname
          : `riskmanagement_${inspectionId}_${Date.now()}.pdf`;

      const finalPath = path.join(directoryPath, fileName);
      try {
        fs.renameSync(req.file.path, finalPath);
      } catch (moveError) {
        console.error('❌ [Risk Management] Error moving PDF, attempting copy:', moveError);
        fs.copyFileSync(req.file.path, finalPath);
        fs.unlinkSync(req.file.path);
      }

      const downloadUrl = `/risk-management/download-pdf/${adminId}/${fileName}`;

      const setData = {
        pdfDownloadUrl: downloadUrl,
        pdfFileName: fileName,
        'step3.pdfDownloadUrl': downloadUrl,
        'step3.pdfFileName': fileName,
        'step3.updatedAt': new Date(),
      };

      const record = await upsertRiskManagementRecord({
        adminId,
        inspectionId,
        setData,
        createdBy: userId,
      });

      res.json({
        success: true,
        fileName,
        filePath: finalPath,
        downloadUrl,
        data: record,
        message: 'Risk management PDF uploaded successfully',
      });
    } catch (error) {
      console.error('❌ [Risk Management] PDF upload error:', error);
      if (req.file && fs.existsSync(req.file.path)) {
        fs.unlinkSync(req.file.path);
      }
      res.status(500).json({
        success: false,
        error: 'Failed to upload risk management PDF: ' + error.message,
      });
    }
  }
);

app.get('/risk-management/download-pdf/:adminId/:fileName', (req, res) => {
  try {
    const { adminId, fileName } = req.params;
    const filePath = path.join(
      __dirname,
      'uploads',
      'documents',
      'riskmanagement',
      adminId || 'unknown',
      fileName
    );

    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'Risk management PDF not found',
      });
    }

    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`,
    });

    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('❌ [Risk Management] PDF download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download risk management PDF',
      });
    });
  } catch (error) {
    console.error('❌ [Risk Management] Download handler error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download risk management PDF: ' + error.message,
    });
  }
});

app.listen(PORT, '0.0.0.0', () => {
  console.log(`⚠️ Risk Assessment server running on port ${PORT}`);
  console.log(
    `⚠️ Risk API base: https://api.titandrillingzm.com:6031/risk-assessments`
  );
});

module.exports = app;

