const express = require('express');
const { mongoose } = require('../dbConnection'); // Use shared connection
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [User Creation Notifications] Using shared MongoDB connection');

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [UserNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [UserNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },

});

// Helper function to send push notifications to all ProAdmins
const sendPushToProAdmins = async (userData, createdByName) => {
  try {
    console.log('📱 Sending push notifications to all ProAdmins about new user...');
    
    const notificationData = {
      roles: ['ProAdmin'],
      title: '👤 New User Created',
      body: `${userData.name} (${userData.role}) has been added to the system\n\nCreated by: ${createdByName}`,
      data: {
        type: 'user_created',
        action: 'view_user',
        userId: userData._id || userData.id,
        userName: userData.name,
        userRole: userData.role,
        userEmail: userData.email || '',
        createdBy: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '👤 New User Created',
              body: `${userData.name} (${userData.role}) has been added\n\nCreated by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'user_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5016,
        path: '/push-notifications/send-to-roles',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} ProAdmin device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (userData, createdByName, proAdmins) => {
  try {
    console.log('💾 Saving user creation notifications to MongoDB for all ProAdmins...');
    
    const notifications = proAdmins.map(admin => ({
      notificationId: `user_created_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${admin._id}`,
      userId: admin._id,
      title: '👤 New User Created',
      body: `${userData.name} (${userData.role}) has been added to the system\n\nCreated by: ${createdByName}`,
      type: 'user_created',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'user_created',
        action: 'view_user',
        userId: userData._id || userData.id,
        userName: userData.name,
        userRole: userData.role,
        userEmail: userData.email || '',
        country: userData.country || '',
        project: userData.project || '',
        createdBy: createdByName,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails to ProAdmins
const sendEmailsToProAdmins = async (userData, createdByName) => {
  try {
    console.log('📧 Sending email notifications to all ProAdmins...');
    
    // Get all ProAdmins with email addresses
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin',
      email: { $exists: true, $ne: null, $ne: '' }
    }).select('name email');
    
    if (proAdmins.length === 0) {
      console.log('⚠️ No ProAdmins with email addresses found');
      return { success: false, error: 'No ProAdmins with emails' };
    }
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s) with email addresses`);
    
    let successCount = 0;
    let failCount = 0;
    
    // Email template for ProAdmins
    const emailSubject = `New User Created: ${userData.name}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>New User Created - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">👤 New User Created</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear Admin,</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">A new user has been added to the Titan Drilling system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Name:</th>
            <td style="padding: 10px; text-align: left;">${userData.name}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Role:</th>
            <td style="padding: 10px; text-align: left;">${userData.role}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Email:</th>
            <td style="padding: 10px; text-align: left;">${userData.email || 'Not provided'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${userData.country || 'Not specified'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${userData.project || 'Not specified'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Created By:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        This user has been granted access to the Titan Drilling application.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Send email to each ProAdmin
    for (const admin of proAdmins) {
      try {
        await emailTransporter.sendMail({
          from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
          to: admin.email,
          subject: emailSubject,
          html: emailHtml,
        });
        
        console.log(`✅ Email sent to ${admin.name} (${admin.email})`);
        successCount++;
      } catch (emailError) {
        console.error(`❌ Failed to send email to ${admin.name}:`, emailError.message);
        failCount++;
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount, total: proAdmins.length };
  } catch (error) {
    console.error('❌ Error sending emails to ProAdmins:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send welcome email to new user
const sendWelcomeEmailToUser = async (userData, createdByName) => {
  try {
    if (!userData.email) {
      console.log('⚠️ No email address for new user, skipping welcome email');
      return { success: false, error: 'No email address' };
    }
    
    console.log(`📧 Sending welcome email to new user: ${userData.name} (${userData.email})`);
    
    const emailSubject = `Welcome to Titan Drilling - ${userData.name}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Welcome to Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">🎉 Welcome to Titan Drilling!</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear ${userData.name},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">Your account has been successfully created in the Titan Drilling system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Name:</th>
            <td style="padding: 10px; text-align: left;">${userData.name}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Role:</th>
            <td style="padding: 10px; text-align: left;">${userData.role}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Email:</th>
            <td style="padding: 10px; text-align: left;">${userData.email}</td>
          </tr>
          ${userData.country ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${userData.country}</td>
          </tr>` : ''}
          ${userData.project ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${userData.project}</td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Created By:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <h3 style="color: #003366; margin-top: 30px;">Next Steps:</h3>
      <ul style="color: #333; line-height: 1.8;">
        <li>Download the Titan Drilling mobile application</li>
        <li>Log in with your credentials</li>
        <li>Complete your profile information</li>
        <li>Start using the system</li>
      </ul>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    await emailTransporter.sendMail({
      from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
      to: userData.email,
      subject: emailSubject,
      html: emailHtml,
    });
    
    console.log(`✅ Welcome email sent to ${userData.name} (${userData.email})`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending welcome email to user:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/user-notifications/new-user
 * Handle notifications when a new user is created
 */
app.post('/api/user-notifications/new-user', async (req, res) => {
  try {
    const { userData, createdByName } = req.body;
    
    if (!userData || !userData.name) {
      return res.status(400).json({
        success: false,
        error: 'User data is required'
      });
    }
    
    console.log(`\n🔔 Processing notifications for new user: ${userData.name}`);
    console.log(`📋 User role: ${userData.role}`);
    console.log(`👤 Created by: ${createdByName}`);
    
    // Get all ProAdmins
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    if (proAdmins.length === 0) {
      console.log('⚠️ No ProAdmins found');
      return res.json({
        success: true,
        message: 'No ProAdmins to notify',
        pushSent: 0,
        emailsSent: 0,
        notificationsSaved: 0,
      });
    }
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s) to notify`);
    
    // Execute all notification tasks in parallel
    const [mongoResult, pushResult, emailResult, welcomeEmailResult] = await Promise.all([
      saveNotificationsToMongoDB(userData, createdByName, proAdmins),
      sendPushToProAdmins(userData, createdByName),
      sendEmailsToProAdmins(userData, createdByName),
      sendWelcomeEmailToUser(userData, createdByName),
    ]);
    
    // Log results
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 ProAdmin Emails: ${emailResult.success ? '✅' : '❌'} (${emailResult.sent || 0} sent)`);
    console.log(`  📧 Welcome Email: ${welcomeEmailResult.success ? '✅' : '❌'}\n`);
    
    res.json({
      success: true,
      message: `Notifications sent for new user: ${userData.name}`,
      results: {
        mongodb: {
          success: mongoResult.success,
          saved: mongoResult.count || 0,
          error: mongoResult.error || null,
        },
        push: {
          success: pushResult.success,
          sent: pushResult.successCount || 0,
          failed: pushResult.failureCount || 0,
          error: pushResult.error || null,
        },
        proAdminEmails: {
          success: emailResult.success,
          sent: emailResult.sent || 0,
          failed: emailResult.failed || 0,
          total: emailResult.total || 0,
          error: emailResult.error || null,
        },
        welcomeEmail: {
          success: welcomeEmailResult.success,
          error: welcomeEmailResult.error || null,
        },
      },
    });
  } catch (error) {
    console.error('❌ Error processing user notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'User Creation Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.USER_NOTIFICATIONS_PORT || 5018;
app.listen(PORT, () => {
  console.log(`\n🚀 User Creation Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: https://api.titandrillingzm.com:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

