const express = require('express');
const { mongoose } = require('../dbConnection'); // Use shared connection
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [H S E Inspection Notifications] Using shared MongoDB connection');

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  projects: [String],
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [HSEInspectionNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [HSEInspectionNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },

});

// Helper function to send push notifications
const sendPushNotifications = async (hseInspectionData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for HSE inspection report...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '🏥 New HSE Inspection Report',
      body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\nProject: ${hseInspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      data: {
        type: 'hse_inspection_report',
        action: 'view_hse_inspection',
        inspectionId: hseInspectionData._id || hseInspectionData.id,
        requestNumber: hseInspectionData.requestNumber,
        checklistName: hseInspectionData.checklistName || '',
        equipmentName: hseInspectionData.equipmentName || '',
        project: hseInspectionData.project || '',
        country: hseInspectionData.country || '',
        inspector: createdByName,
        safePercentage: hseInspectionData.percentageSafe || 0,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '🏥 New HSE Inspection Report',
              body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\n\nInspected by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'hse_inspection_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (hseInspectionData, createdByName, recipients) => {
  try {
    console.log('💾 Saving HSE inspection notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `hse_insp_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '🏥 New HSE Inspection Report',
      body: `${hseInspectionData.requestNumber}\nChecklist: ${hseInspectionData.checklistName || 'N/A'}\nProject: ${hseInspectionData.project || 'N/A'}\n\nInspected by: ${createdByName}`,
      type: 'hse_inspection_report',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'hse_inspection_report',
        action: 'view_hse_inspection',
        inspectionId: hseInspectionData._id || hseInspectionData.id,
        requestNumber: hseInspectionData.requestNumber,
        checklistName: hseInspectionData.checklistName || '',
        equipmentName: hseInspectionData.equipmentName || '',
        project: hseInspectionData.project || '',
        country: hseInspectionData.country || '',
        inspector: createdByName,
        safePercentage: hseInspectionData.percentageSafe || 0,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails
const sendEmails = async (hseInspectionData, createdByName, recipients, pdfUrl) => {
  try {
    console.log('📧 Sending HSE inspection emails...');
    console.log(`📧 Recipients: ${recipients.length}`);
    console.log(`📧 PDF URL: ${pdfUrl}`);
    
    let successCount = 0;
    let failCount = 0;
    
    for (const recipient of recipients) {
      if (recipient.email && recipient.email.includes('@')) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <sonia.owninfotech@gmail.com>',
            to: recipient.email,
            subject: `New HSE Inspection Report: ${hseInspectionData.requestNumber}`,
            html: `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>New HSE Inspection Report - Titan Drilling</title>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0; }
    .wrapper { max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5; }
    .header { background-color: #004a93; padding: 24px; text-align: center; border-bottom: 3px solid #003366; }
    .header img { width: 120px; }
    .content { padding: 32px; color: #333; }
    .content h2 { color: #003366; margin-bottom: 20px; font-size: 22px; }
    .summary { margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 18px; }
    .summary table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    .summary th { width: 35%; text-align: left; padding: 10px; color: #003366; font-weight: 600; }
    .summary td { padding: 10px; text-align: left; color: #333; }
    .footer { background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555; }
    .cta { display: inline-block; margin-top: 24px; padding: 12px 28px; background: #004a93; color: #ffffff; text-decoration: none; border-radius: 24px; font-weight: 600; }
  </style>
</head>
<body>
  <div class="wrapper">
    <div class="header">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" />
    </div>
    <div class="content">
      <h2>New HSE Inspection Report</h2>
      <p style="line-height: 1.6; margin-bottom: 16px;">Dear ${recipient.name || 'Team Member'},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">A new HSE inspection report has been submitted in the Titan Drilling system.</p>

      <div class="summary">
        <table>
          <tr>
            <th>Request Number</th>
            <td>${hseInspectionData.requestNumber}</td>
          </tr>
          <tr>
            <th>Checklist</th>
            <td>${hseInspectionData.checklistName || 'N/A'}</td>
          </tr>
          ${hseInspectionData.equipmentName ? `
          <tr>
            <th>Equipment</th>
            <td>${hseInspectionData.equipmentName}</td>
          </tr>` : ''}
          <tr>
            <th>Country</th>
            <td>${hseInspectionData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th>Project</th>
            <td>${hseInspectionData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th>Inspector</th>
            <td>${createdByName}</td>
          </tr>
          ${hseInspectionData.percentageSafe !== undefined ? `
          <tr>
            <th>Safety Score</th>
            <td>${hseInspectionData.percentageSafe}% Safe</td>
          </tr>` : ''}
          <tr>
            <th>Date</th>
            <td>${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 24px;">
        Please review the report in the Titan Drilling application.${pdfUrl ? ' You can also download the PDF using the link below.' : ''}
      </p>

      ${pdfUrl ? `<a class="cta" href="${pdfUrl}">Download PDF Report</a>` : ''}

      <p style="line-height: 1.6; margin-top: 24px;">
        If you require assistance, please contact
        <a href="mailto:support@titandrillingzm.com" style="color: #003366; font-weight: 600; text-decoration: none;">
          support@titandrillingzm.com
        </a>.
      </p>

      <p style="line-height: 1.6; margin-top: 16px;">Best regards,<br /><strong>Titan Drilling Team</strong></p>
    </div>
    <div class="footer">
      © ${new Date().getFullYear()} Titan Drilling. All rights reserved.
    </div>
  </div>
</body>
</html>
            `,
          };
          
          await emailTransporter.sendMail(mailOptions);
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email Summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send confirmation email to inspector
const sendInspectorConfirmationEmail = async (hseInspectionData, inspectorName, inspectorEmail, pdfUrl) => {
  try {
    if (!inspectorEmail || !inspectorEmail.includes('@')) {
      console.log('⚠️ No valid inspector email provided');
      return { success: false, error: 'No valid email' };
    }
    
    console.log(`📧 Sending confirmation email to inspector: ${inspectorEmail}`);
    
    const mailOptions = {
      from: '"Titan Drilling" <sonia.owninfotech@gmail.com>',
      to: inspectorEmail,
      subject: `HSE Inspection Report Submitted: ${hseInspectionData.requestNumber}`,
      html: `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>HSE Inspection Report Submitted - Titan Drilling</title>
  <style>
    body { font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0; }
    .wrapper { max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5; }
    .header { background-color: #004a93; padding: 24px; text-align: center; border-bottom: 3px solid #003366; }
    .header img { width: 120px; }
    .content { padding: 32px; color: #333; }
    .content h2 { color: #003366; margin-bottom: 20px; font-size: 22px; }
    .summary { margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 18px; }
    .summary table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    .summary th { width: 35%; text-align: left; padding: 10px; color: #003366; font-weight: 600; }
    .summary td { padding: 10px; text-align: left; color: #333; }
    .footer { background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555; }
    .cta { display: inline-block; margin-top: 24px; padding: 12px 28px; background:rgb(235, 235, 235); color: #ffffff; text-decoration: none; border-radius: 24px; font-weight: 600; }
  </style>
</head>
<body>
  <div class="wrapper">
    <div class="header">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" />
    </div>
    <div class="content">
      <h2>HSE Inspection Report Submitted</h2>
      <p style="line-height: 1.6; margin-bottom: 16px;">Dear ${inspectorName || 'Inspector'},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">Your HSE inspection report has been submitted successfully. Relevant supervisors and managers have been notified.</p>

      <div class="summary">
        <table>
          <tr>
            <th>Request Number</th>
            <td>${hseInspectionData.requestNumber}</td>
          </tr>
          <tr>
            <th>Checklist</th>
            <td>${hseInspectionData.checklistName || 'N/A'}</td>
          </tr>
          ${hseInspectionData.equipmentName ? `
          <tr>
            <th>Equipment</th>
            <td>${hseInspectionData.equipmentName}</td>
          </tr>` : ''}
          <tr>
            <th>Country</th>
            <td>${hseInspectionData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th>Project</th>
            <td>${hseInspectionData.project || 'N/A'}</td>
          </tr>
          ${hseInspectionData.percentageSafe !== undefined ? `
          <tr>
            <th>Safety Score</th>
            <td>${hseInspectionData.percentageSafe}% Safe</td>
          </tr>` : ''}
          <tr>
            <th>Submitted</th>
            <td>${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      ${pdfUrl ? `<a class="cta" href="${pdfUrl}">Download Your PDF Report</a>` : ''}

      <p style="line-height: 1.6; margin-top: 24px;">
        Thank you for maintaining our safety standards. If you require assistance, contact
        <a href="mailto:support@titandrillingzm.com" style="color: #003366; font-weight: 600; text-decoration: none;">
          support@titandrillingzm.com
        </a>.
      </p>
    </div>
    <div class="footer">
      © ${new Date().getFullYear()} Titan Drilling. All rights reserved.
    </div>
  </div>
</body>
</html>
      `,
    };
    
    await emailTransporter.sendMail(mailOptions);
    console.log(`✅ Confirmation email sent to inspector: ${inspectorEmail}`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending inspector confirmation email:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/hse-inspection-notifications/new-inspection
 * Handle notifications when a new HSE inspection report is created
 */
app.post('/api/hse-inspection-notifications/new-inspection', async (req, res) => {
  try {
    console.log('\n🔔 ========== NEW HSE INSPECTION NOTIFICATION REQUEST ==========');
    console.log('📥 Received payload:', JSON.stringify(req.body, null, 2));
    
    const { hseInspectionData, createdByName, inspectorEmail, pdfUrl } = req.body;
    
    if (!hseInspectionData || !hseInspectionData.requestNumber) {
      console.error('❌ Missing HSE inspection data or request number');
      return res.status(400).json({
        success: false,
        error: 'HSE inspection data with request number is required'
      });
    }
    
    console.log(`\n🔔 Processing notifications for HSE inspection: ${hseInspectionData.requestNumber}`);
    console.log(`📋 Country: ${hseInspectionData.country}, Project: ${hseInspectionData.project}`);
    console.log(`👤 Inspector: ${createdByName}`);
    console.log(`📧 Inspector Email: ${inspectorEmail}`);
    console.log(`📄 PDF URL: ${pdfUrl}`);
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      $or: [
        { country: hseInspectionData.country, project: hseInspectionData.project },
        { countries: hseInspectionData.country, projects: hseInspectionData.project }
      ]
    }).select('_id name email fcmToken fcmTokens role country countries project projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${hseInspectionData.country} - ${hseInspectionData.project}`);
    
    // Get Managers filtered by country AND project
    const managerRoles = ['HrManager', 'Workshop Manager', 'Maintenance Manager', 'FinanceManager', 'Project Manager', 'Site Manager'];
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      $or: [
        { country: hseInspectionData.country, project: hseInspectionData.project },
        { countries: hseInspectionData.country, projects: hseInspectionData.project }
      ]
    }).select('_id name email fcmToken fcmTokens role country countries project projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${hseInspectionData.country} - ${hseInspectionData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found for notifications');
    }
    
    // Execute all notification tasks in parallel
    console.log('🚀 Starting parallel execution of notifications...');
    
    const results = await Promise.all([
      saveNotificationsToMongoDB(hseInspectionData, createdByName, uniqueRecipients),
      sendPushNotifications(hseInspectionData, createdByName, uniqueRecipients),
      sendEmails(hseInspectionData, createdByName, uniqueRecipients, pdfUrl),
      sendInspectorConfirmationEmail(hseInspectionData, createdByName, inspectorEmail, pdfUrl)
    ]);
    
    const [mongoResult, pushResult, emailResult, inspectorEmailResult] = results;
    
    // Log results
    console.log('\n📊 ========== NOTIFICATION RESULTS ==========');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails: ${emailResult.sent || 0} sent, ${emailResult.failed || 0} failed`);
    console.log(`  📧 Inspector Email: ${inspectorEmailResult.success ? '✅' : '❌'}`);
    console.log('========================================\n');
    
    res.json({
      success: true,
      message: `Notifications sent for HSE inspection: ${hseInspectionData.requestNumber}`,
      results: {
        mongodb: mongoResult,
        push: pushResult,
        emails: emailResult,
        inspectorEmail: inspectorEmailResult
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length
      }
    });
  } catch (error) {
    console.error('❌ Error processing HSE inspection notifications:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'HSE Inspection Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.HSE_INSPECTION_NOTIFICATIONS_PORT || 5028;
app.listen(PORT, () => {
  console.log(`\n🚀 HSE Inspection Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: https://api.titandrillingzm.com:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

