import { View, Text, TextInput, ScrollView, StyleSheet, TouchableOpacity, Dimensions, Alert, Modal, Image, Animated } from 'react-native'
import React, { useState, useRef, useEffect } from 'react'
import MaterialIcons from 'react-native-vector-icons/MaterialIcons'
import { useNavigation } from '@react-navigation/native'
import AsyncStorage from '@react-native-async-storage/async-storage'
import SignatureCanvas from 'react-native-signature-canvas'
import database from '@react-native-firebase/database'
import storage from '@react-native-firebase/storage'
import NetInfo from '@react-native-community/netinfo'
import RNFS from 'react-native-fs'
import RNHTMLtoPDF from 'react-native-html-to-pdf'
import Toast from 'react-native-toast-message'

const { width, height } = Dimensions.get('window')

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_077_078Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation()
  const [formData, setFormData] = useState({
    correctiveAction: '',
    furtherInvestigation: '',
    supervisorName: '',
    supervisorSignature: ''
  })

  const [isLoading, setIsLoading] = useState(false)
  const [showSuccessModal, setShowSuccessModal] = useState(false)
  const [reportId, setReportId] = useState(null)
  const [signatureModalVisible, setSignatureModalVisible] = useState(false)
  const [supervisorSignatureData, setSupervisorSignatureData] = useState(null)
  const [step1Data, setStep1Data] = useState(null)
  const [step2Data, setStep2Data] = useState(null)
  const isMounted = useRef(true)
  const signatureRef = useRef(null)

  useEffect(() => {
    loadData()
    isMounted.current = true
    return () => {
      isMounted.current = false
    }
  }, [])

  const loadData = async () => {
    try {
      const savedStep1 = await AsyncStorage.getItem('hseStep1')
      const savedStep2 = await AsyncStorage.getItem('hseStep2')
      
      if (savedStep1) {
        setStep1Data(JSON.parse(savedStep1))
      }
      if (savedStep2) {
        setStep2Data(JSON.parse(savedStep2))
      }
    } catch (error) {
      console.error('Error loading data:', error)
    }
  }

  const updateFormData = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }))
  }

  const selectInvestigation = (option) => {
    setFormData(prev => ({
      ...prev,
      furtherInvestigation: option
    }))
  }

  const handleBackPress = () => {
    setActiveStep(2)
  }

  const openSignatureModal = () => {
    setSignatureModalVisible(true)
  }

  const closeSignatureModal = () => {
    setSignatureModalVisible(false)
  }

  const handleSignatureComplete = (signature) => {
    if (signature) {
      setSupervisorSignatureData(signature)
      setFormData(prev => ({
        ...prev,
        supervisorSignature: signature
      }))
    }
    setSignatureModalVisible(false)
  }

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature()
    }
  }

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature()
    }
  }

  const handleSignatureData = (signature) => {
    console.log('Signature captured:', signature ? signature.substring(0, 50) + '...' : signature)
    handleSignatureComplete(signature)
  }

  // Firebase helper functions
  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail')
    if (!userEmail) throw new Error('User email not found. Please log in again.')
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value')
    const userData = userSnapshot.val()
    if (!userData) throw new Error('User not found in admin database.')
    return Object.keys(userData)[0]
  }

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return (currentNumber + 1).toString().padStart(3, '0');
    } catch (error) {
      console.error('Error getting current report number:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val();
      
      // Initialize counter if it doesn't exist
      if (currentNumber === null || currentNumber === undefined) {
        currentNumber = 0;
      }
      
      currentNumber++;
      
      // Update the counter in Firebase
      await counterRef.set(currentNumber);
      
      return currentNumber.toString().padStart(3, '0');
    } catch (error) {
      console.error('Error incrementing report counter:', error);
      return '001';
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath)
      await reference.putFile(filePath)
      const downloadUrl = await reference.getDownloadURL()
      return downloadUrl
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error)
      throw error
    }
  }

  const flushStepData = async () => {
    try {
      await AsyncStorage.removeItem('hseStep1')
      await AsyncStorage.removeItem('hseStep2')
    } catch (error) {
      console.error('Error flushing step data:', error)
      throw new Error('Failed to clear local data')
    }
  }

  const validateForm = () => {
    if (!formData.correctiveAction.trim()) {
      Alert.alert('Missing Information', 'Please describe the corrective action taken.')
      return false
    }
    
    if (!formData.furtherInvestigation) {
      Alert.alert('Missing Information', 'Please select whether further investigation is needed.')
      return false
    }
    
    if (!formData.supervisorName.trim()) {
      Alert.alert('Missing Information', 'Please enter the supervisor name.')
      return false
    }
    
    if (!formData.supervisorSignature) {
      Alert.alert('Missing Information', 'Please provide the supervisor signature.')
      return false
    }
    
    return true
  }

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {}
      const step2 = report.step2 || {}
      const step3 = report.step3 || {}
      const checklist = report.checklist || {}

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Near Miss / Hazard Identification Slip</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
              padding: 0;
              font-size: 12px;
              line-height: 1.4;
            }
            .header-row {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 20px;
            }
            .logo-left {
              display: flex;
              align-items: center;
            }
            .logo-left img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .company-name {
              font-size: 18px;
              font-weight: bold;
              color: #015185;
              margin-left: 10px;
            }
            .doc-info-box {
              width: 340px;
            }
            .doc-info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .doc-info-box td {
              border: 1px solid #222;
              padding: 4px 6px;
              text-align: left;
            }
            .title-section {
              text-align: center;
              margin: 20px 0;
            }
            .title-section h2 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              text-decoration: underline;
              text-transform: uppercase;
            }
            .slip-number {
              text-align: right;
              margin-top: 10px;
              font-weight: bold;
            }
            .form-section {
              margin: 15px 0;
            }
            .form-row {
              display: flex;
              margin-bottom: 10px;
              align-items: flex-start;
            }
            .form-label {
              font-weight: bold;
              min-width: 200px;
              margin-right: 10px;
            }
            .form-value {
              flex: 1;
              border-bottom: 1px solid #000;
              min-height: 20px;
              padding: 2px 5px;
            }
            .form-textarea {
              flex: 1;
              border: 1px solid #000;
              min-height: 80px;
              padding: 5px;
              margin-top: 5px;
            }
            .hazard-class-section {
              margin: 15px 0;
            }
            .hazard-class-label {
              font-weight: bold;
              margin-bottom: 10px;
            }
            .hazard-class-options {
              display: flex;
              gap: 20px;
            }
            .hazard-class-option {
              display: flex;
              align-items: center;
            }
            .hazard-class-option input[type="radio"] {
              margin-right: 5px;
            }
            .instruction-section {
              text-align: center;
              margin-top: 30px;
              padding: 15px;
              background-color: #f8f9fa;
              border: 1px solid #e9ecef;
            }
            .instruction-text {
              font-size: 14px;
              font-weight: bold;
              color: #dc3545;
              font-style: italic;
              text-transform: uppercase;
            }
            .signature-section {
              margin-top: 20px;
            }
            .signature-container {
              display: flex;
              justify-content: space-between;
              margin-top: 10px;
            }
            .signature-box {
              width: 48%;
              border: 1px solid #000;
              padding: 10px;
              min-height: 60px;
            }
            .signature-label {
              font-weight: bold;
              margin-bottom: 5px;
            }
            .signature-image {
              text-align: center;
              margin-top: 10px;
            }
            .signature-image img {
              max-width: 150px;
              max-height: 60px;
              border: 1px solid #ccc;
            }
          </style>
        </head>
        <body>
          <div class="header-row">
            <div class="logo-left">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
              <div class="company-name">TITAN DRILLING</div>
            </div>
            <div class="doc-info-box">
              <table>
                <tr><td>Document Name</td><td>HSEC-FOR-6.1.2 Near Miss & Hazard Identification Slip</td></tr>
                <tr><td>Document ID</td><td>HSE-078</td></tr>
                <tr><td>Version Date</td><td>03/08/2021</td></tr>
                <tr><td>Version</td><td>1</td></tr>
                <tr><td>Approved</td><td>HSM</td></tr>
                <tr><td>Request #</td><td>${step1.requestNumber || 'N/A'}</td></tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h2>NEAR MISS / HAZARD IDENTIFICATION SLIP</h2>
            <div class="slip-number">No: ${report.id || 'N/A'}</div>
          </div>

          <div class="form-section">
            <div class="form-row">
              <div class="form-label">Date:</div>
              <div class="form-value">${step2.date || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Country:</div>
              <div class="form-value">${step1.country || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Project:</div>
              <div class="form-value">${step1.project || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Name of Person Reporting:</div>
              <div class="form-value">${step2.name || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Dept:</div>
              <div class="form-value">${step2.department || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">NM / Hazard Description:</div>
              <div class="form-textarea">${step2.description || ''}</div>
            </div>
            
            <div class="hazard-class-section">
              <div class="hazard-class-label">Hazard Class A B C (circle):</div>
              <div class="hazard-class-options">
                <div class="hazard-class-option">
                  <input type="radio" ${step2.hazardClass === 'A' ? 'checked' : ''} disabled> A
                </div>
                <div class="hazard-class-option">
                  <input type="radio" ${step2.hazardClass === 'B' ? 'checked' : ''} disabled> B
                </div>
                <div class="hazard-class-option">
                  <input type="radio" ${step2.hazardClass === 'C' ? 'checked' : ''} disabled> C
                </div>
              </div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Area (location):</div>
              <div class="form-value">${step2.location || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">What corrective action taken:</div>
              <div class="form-textarea">${step3.correctiveAction || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Further investigation:</div>
              <div class="form-value">${step3.furtherInvestigation || ''}</div>
            </div>
            
            <div class="signature-section">
              <div class="form-label">Signature Supervisor:</div>
              <div class="signature-container">
                <div class="signature-box">
                  <div class="signature-label">Name: ${step3.supervisorName || ''}</div>
                  <div class="signature-image">
                    ${step3.supervisorSignature ? `<img src="${step3.supervisorSignature}" alt="Supervisor Signature"/>` : ''}
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div class="instruction-section">
            <div class="instruction-text">
              HAND INTO SAFETY DEPARTMENT FOR CAPTURING AND FOLLOW UP
            </div>
          </div>
        </body>
        </html>
      `

      const options = {
        html,
        fileName: report.id,
        directory: 'Documents',
      }

      const file = await RNHTMLtoPDF.convert(options)
      return file.filePath
    } catch (error) {
      console.error('Error generating PDF:', error)
      throw error
    }
  }

  const handleSubmit = async () => {
    if (!validateForm()) {
      return
    }

    setIsLoading(true)

    try {
      const userId = await getAdminUserId()
      const savedStep1 = await AsyncStorage.getItem('hseStep1')
      const savedStep2 = await AsyncStorage.getItem('hseStep2')
      
      if (!savedStep2) {
        throw new Error('Step 2 data is missing. Please complete Step 2 first.')
      }

      // For HSE_077_078, we might not have Step1 data, so create a default structure
      let step1Data = {}
      if (savedStep1) {
        step1Data = JSON.parse(savedStep1)
      } else {
        // Create default Step1 data structure for HSE_077_078
        step1Data = {
          selectedChecklist: {
            document_name: 'HSEC-FOR-6.1.2 Near Miss & Hazard Identification Slip',
            document_id: 'HSE-078',
            version_date: '03/08/2021',
            version: '1',
            approved_by: 'HSM'
          }
        }
      }

      const step2Data = JSON.parse(savedStep2)

      // Use the request number from step 1
      const requestId = step1Data.requestNumber
      if (!requestId) {
        throw new Error('Request number not found in step 1 data')
      }

      // Store only the essential filled data from PDF
      const pdfData = {
        // Basic report info
        report_id: requestId,
        request_number: requestId,
        
        // Only filled form data from step 2 and 3
        date: step2Data.date || '',
        name: step2Data.name || '',
        department: step2Data.department || '',
        description: step2Data.description || '',
        hazard_class: step2Data.hazardClass || '',
        location: step2Data.location || '',
        corrective_action: formData.correctiveAction || '',
        further_investigation: formData.furtherInvestigation || '',
        supervisor_name: formData.supervisorName || '',
        supervisor_signature: formData.supervisorSignature || '',
        manager_name: formData.managerName || '',
        manager_signature: formData.managerSignature || '',
      }

      const reportData = {
        id: requestId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1: step1Data,
        step2: step2Data,
        step3: formData,
        pdfData, // Store only essential PDF data
      }

      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${requestId}`)
      
      // Save report metadata first
      await dbRef.set({
        id: requestId,
        userId,
        createdAt: reportData.createdAt,
        status: 'uploading',
        pdfData, // Store PDF data immediately
      })

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(reportData)
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/${requestId}.pdf`
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath)

      // Update report with PDF URL and final status
      await dbRef.update({
        pdfDownloadUrl,
        status: 'completed',
      })

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err))
      
      // Increment the global report counter only after successful report generation
      await incrementReportCounter()
      
      await flushStepData()

      setIsLoading(false)
      setShowSuccessModal(true)
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Near Miss & Hazard Identification Report submitted successfully.',
        position: 'top',
      })
    } catch (error) {
      console.error('Error in handleSubmit:', error)
      setIsLoading(false)
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to submit report.',
        position: 'top',
      })
      Alert.alert('Submit Error', `Failed to submit report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false })
    }
  }

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.')
      return
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail')
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.')
        navigation.replace('AdminLoginScreen')
        return
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value')
      const userData = userSnapshot.val()
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.')
        navigation.replace('AdminLoginScreen')
        return
      }
      const userId = Object.keys(userData)[0]

      // Use the same database path as Hsemodule43step3
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`)
      const snapshot = await reportRef.once('value')
      const reportData = snapshot.val()

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.')
        return
      }

      const pdfUrl = reportData.pdfDownloadUrl
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.')
        return
      }

      navigation.navigate('HSEViewPDF', { 
        pdfUrl: pdfUrl, 
        onGoBack: () => { 
          setShowSuccessModal(false) 
        } 
      })
    } catch (error) {
      console.error('Error fetching PDF URL:', error)
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.')
    } finally {
      setShowSuccessModal(false)
    }
  }

  const canvasStyle = `
    .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
    .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
    canvas { height: 100% !important; width: 100% !important; }
  `

  return (
    <View style={styles.container}>
      <ScrollView 
        style={styles.scrollView}
        contentContainerStyle={styles.scrollContent}
        showsVerticalScrollIndicator={true}
        keyboardShouldPersistTaps="handled"
      >
        <View style={styles.formContainer}>
          <Text style={styles.title}>Corrective Action & Follow-up</Text>
          
          {/* Corrective Action Field */}
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>What corrective action taken:</Text>
            <TextInput
              style={[styles.input, styles.textArea]}
              value={formData.correctiveAction}
              onChangeText={(text) => updateFormData('correctiveAction', text)}
              placeholder="Describe the corrective actions taken..."
              multiline
              numberOfLines={4}
              textAlignVertical="top"
            />
          </View>

          {/* Further Investigation Field */}
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>Further investigation:</Text>
            <View style={styles.investigationContainer}>
              <TouchableOpacity
                style={[
                  styles.investigationButton,
                  formData.furtherInvestigation === 'Yes' && styles.selectedInvestigation
                ]}
                onPress={() => selectInvestigation('Yes')}
              >
                <Text style={[
                  styles.investigationText,
                  formData.furtherInvestigation === 'Yes' && styles.selectedInvestigationText
                ]}>Yes</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.investigationButton,
                  formData.furtherInvestigation === 'No' && styles.selectedInvestigation
                ]}
                onPress={() => selectInvestigation('No')}
              >
                <Text style={[
                  styles.investigationText,
                  formData.furtherInvestigation === 'No' && styles.selectedInvestigationText
                ]}>No</Text>
              </TouchableOpacity>
            </View>
          </View>

          {/* Supervisor Name Field */}
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>Name of Supervisor:</Text>
            <TextInput
              style={styles.input}
              value={formData.supervisorName}
              onChangeText={(text) => updateFormData('supervisorName', text)}
              placeholder="Enter supervisor name"
            />
          </View>

          {/* Supervisor Signature Field */}
          <View style={styles.fieldContainer}>
            <Text style={styles.label}>Signature Supervisor:</Text>
            {supervisorSignatureData ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: supervisorSignatureData }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={openSignatureModal}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={openSignatureModal}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>

          {/* Instruction Text */}
          <View style={styles.instructionContainer}>
            <Text style={styles.instructionText}>
              HAND INTO SAFETY DEPARTMENT FOR CAPTURING AND FOLLOW UP
            </Text>
          </View>
        </View>
      </ScrollView>
      
      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity 
          style={[styles.submitButton, isLoading && styles.disabledButton]} 
          onPress={handleSubmit}
          disabled={isLoading}
        >
          <Text style={styles.submitButtonText}>Submit</Text>
          <MaterialIcons style={styles.bottomicon} name="check" size={16} color="#fff" />
        </TouchableOpacity>
      </View>

      {/* SyncModal for Loading */}
      <SyncModal visible={isLoading} message="Generating Report" />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              Near Miss / Hazard Identification Slip Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false)
                  navigation.navigate('HSEInspectionHome')
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Signature Modal */}
      <Modal
        visible={signatureModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>Supervisor Signature</Text>
              <TouchableOpacity onPress={closeSignatureModal}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty')
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={canvasStyle}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error)
                  Alert.alert('Error', 'Failed to load signature canvas.')
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </View>
  )
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 120, // Add extra padding to ensure content is not hidden behind buttons
  },
  formContainer: {
    padding: 20,
    backgroundColor: 'white',
    margin: 10,
    borderRadius: 10,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    elevation: 5,
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 20,
    color: '#333',
  },
  fieldContainer: {
    marginBottom: 25,
  },
  label: {
    fontSize: 16,
    fontWeight: '600',
    marginBottom: 8,
    color: '#333',
  },
  input: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  textArea: {
    height: 100,
    textAlignVertical: 'top',
  },
  investigationContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 10,
  },
  investigationButton: {
    width: 80,
    height: 40,
    borderRadius: 20,
    borderWidth: 2,
    borderColor: '#ddd',
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  selectedInvestigation: {
    borderColor: '#007AFF',
    backgroundColor: '#007AFF',
  },
  investigationText: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
  },
  selectedInvestigationText: {
    color: '#fff',
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
  },
  signatureImage: {
    width: width * 0.5,
    height: height * 0.12,
    borderRadius: 10,
    backgroundColor: '#f5f5f5',
  },
  resignButton: {
    backgroundColor: '#015185',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
  },
  resignButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  signatureBox: {
    height: height * 0.15,
    backgroundColor: '#f5f5f5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#ddd',
    borderStyle: 'dashed',
  },
  signaturePlaceholder: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  instructionContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    marginTop: 10,
  },
  instructionText: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#dc3545',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  submitButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 20,
    backgroundColor: '#28a745',
    borderRadius: 8,
  },
  submitButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginRight: 5,
  },
  disabledButton: {
    opacity: 0.5,
  },
  bottomicon: {
    marginTop: 2,
  },
  // Modal Styles
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '95%',
    maxHeight: '90%',
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '60%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
})

export default HSE_077_078Step3
